<?php

declare(strict_types=1);

namespace Inside\Authentication\Http\Controllers\Auth;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Authentication\Contracts\Authentication as AuthenticationContract;
use Inside\Authentication\Http\Controllers\Controller;
use OpenApi\Annotations as OA;

final class Login extends Controller
{
    public function __invoke(Request $request, AuthenticationContract $authentication): JsonResponse
    {
        $credentials = $request->only('name', 'email', 'password', 'captcha_token');
        $remember = (bool) $request->get('remember', false);

        $token = $authentication->login($credentials, $remember);
        $user = $token->accessToken->user;

        $transformed = $user->toArray();
        $transformed['api_token'] = $token->plainTextToken;

        // Set header so we can use auth api guard
        $request->headers->set('api_token', $token->plainTextToken);

        return response()->json(['data' => $transformed]);
    }
    /**
     * Login the user in the application
     *
     * @OA\Post(
     *      path="/authentication/login",
     *      operationId="authenticationLogin",
     *      tags={"Authentification"},
     *      summary="Système d'authentification pour obtenir son token de navigation",
     *     @OA\Parameter(
     *         name="email",
     *         in="query",
     *         description="Email d'autentification",
     *         required=true,
     *         example="technique@maecia.com",
     *         @OA\Schema(
     *             type="string"
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="password",
     *         in="query",
     *         description="Le mot de passe",
     *         required=true,
     *         example="EQKy""TB~z*C2jt.k&",
     *         @OA\Schema(
     *             type="string"
     *         )
     *     ),
     *      @OA\Response(
     *          response=200,
     *          description="Les informations sur l'utilisateur au format JSON",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *           @OA\Schema(
     *            type="object",
     *           @OA\Property(
     *                     property="data",
     *                     type="object",
     *                     description="Les informations de l'utilisateur connecté",
     *                      @OA\Property(
     *                          property="uuid",
     *                          description="UUID du compte utilisateur",
     *                          type="string",
     *                          example="1d4f5c59-46c8-44c1-b939-d63a2e206cf8"
     *                     ),
     *                      @OA\Property(
     *                          property="uuid_host",
     *                          description="UUID drupal",
     *                          type="string",
     *                          deprecated=true,
     *                          example="966fb423-9613-4dec-aa7c-767bdee55586"
     *                     ),
     *                      @OA\Property(
     *                          property="name",
     *                          description="Identifiant de connection",
     *                          type="string",
     *                          example="super_admin"
     *                     ),
     *                      @OA\Property(
     *                          property="email",
     *                          description="Email de compte",
     *                          type="string",
     *                          example="technique@maecia.com"
     *                     ),
     *                      @OA\Property(
     *                          property="status",
     *                          description="Etat de connection ( devrait toujours être 1) car on ne peut pas se
     *                          connecter avec un compte bloqué ou inactif.", type="integer", default=1, example=1
     *                     ),
     *                      @OA\Property(
     *                          property="langcode",
     *                          description="Langue d'interface pour cette utilisateur",
     *                          type="string",
     *                          example="fr"
     *                     ),
     *                      @OA\Property(
     *                          property="api-token",
     *                          description="Le token de connection requis pour tous les appels api securisé",
     *                          type="string",
     *                          example="XgsvchbOgeDxqUtvmZuqxH8mNXxFFFKkf4QUNiD33B49ZpUeapZgfbM1yWka"
     *                     ),
     *                      @OA\Property(
     *                          property="permissions",
     *                          description="Les informations liés aux droits de cet utilisateur",
     *                          type="object",
     *                          @OA\Property(
     *                             property="roles",
     *                             description="Les rôles auxquels appartient cet utilisateur",
     *                             type="array",
     *                             items=@OA\Items(type="string",description="Le nom machine du rôle",
     *                             example="authenticated")
     *                         ),
     *                          @OA\Property(
     *                             property="add",
     *                             description="Les types ( nom machine ) de contenu auxquels cet utilisateur peut
     *                             ajouter du contenu", type="array", items=@OA\Items(type="string",description="Le
     *                             type machine du contenut", example="news")
     *                         ),
     *                     )
     *              )
     *           )
     *       ),
     * @OA\Response(response=400, description="Bad request",@OA\MediaType(
     *           mediaType="application/json",
     *            @OA\schema(type="object",
     *           @OA\Property(property="error",type="string",example="Les informations d'identification sont
     *                                                                    incorrectes"))))), security={}
     *  )
     */
}
