<?php

declare(strict_types=1);

namespace Inside\Authentication\Http\Controllers\Auth;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Facades\Authentication;
use Inside\Authentication\Http\Controllers\Controller;
use Inside\Kernel\Authentication\StoreIntendedPathService;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;

final class ExternalLogin extends Controller
{
    public function __invoke(Request $request): JsonResponse
    {
        $user = Authentication::getUserFromMagicCookie($request);

        if (is_null($user)) {
            Authentication::invalidateUserFromMagicCookie();

            throw new AuthorizationException(Lang::get('forbidden.notAuthorized'));
        }

        $token = Authentication::logAs($user, $user->provider_type ?? 'inside-external', true);

        $transformed = $user->only(['uuid', 'uuid_host', 'langcode', 'email', 'status']);

        $transformed['api_token'] = $token->plainTextToken;
        $transformed['url_intended'] = StoreIntendedPathService::retrieve();
        $transformed['permissions'] = [
            'roles' => Role::listUserRoles($user),
            'add' => Permission::getAllowedCreationTypesForUser($user),
            'custom' => Permission::customList($user),
            'backoffice_entries' => Permission::backofficeAccessibleEntries($user),
        ];

        StoreIntendedPathService::forget();

        $request->headers->set('api_token', $token->plainTextToken);

        return response()->json(['data' => $transformed]);
    }
}
