<?php

declare(strict_types=1);

namespace Inside\Authentication\Http\Controllers\Auth;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Contracts\Authentication as AuthenticationContract;
use Inside\Authentication\Http\Controllers\Controller;
use Inside\Authentication\Models\User;
use Inside\Events\PageVisited;
use Inside\Facades\Package;
use Inside\Host\Bridge\BridgeAuthentication;

final class Check extends Controller
{
    public function __invoke(Request $request, AuthenticationContract $authentication): JsonResponse
    {
        $apiToken = $request->header('api-token');
        $devEnv = (bool) $request->header('dev-env');

        /** @var User $user */
        $user = Auth::user();

        $isConnected = is_string($apiToken)
            && $user instanceof User
            && $user->currentAccessToken()
            && $user->currentAccessToken()->token === $authentication->encryptToken($apiToken);
        if (! $isConnected && Cookie::has('_inside_token')) {
            // Remove old cookie
            $authentication->invalidateUserFromMagicCookie();
        } elseif ($isConnected && ! Cookie::has('_inside_token')) {
            // magic cookie is missing, we should rebuild it !
            $authentication->setUserToMagicCookie($request, $user, $apiToken);
        }

        if ($isConnected && $user->currentAccessToken() && $user->currentAccessToken()->authenticator !== 'inside-admin') {
            try {
                $lastUpdate = $user->last_access_at ?? null;
                $lastLogin = $user->last_login_at ?? null;

                if (! $lastUpdate ||
                    ($lastLogin && $lastLogin->isAfter($lastUpdate)) ||
                    $lastUpdate->diffInMinutes(now()) > 5) {
                    $user->update(['last_access_at' => now()]);
                }
            } catch (\Exception $e) {
                Log::error('[Check] Error updating last access time', [
                    'error' => $e->getMessage(),
                    'last_access_at' => $lastUpdate,
                    'user_uuid' => $user->uuid,
                    'email' => $user->email,
                    'is_connected' => $isConnected,
                    'authenticator' => optional($user->currentAccessToken())->authenticator,
                ]);
            }
        }

        $response = response()->json(['data' => $isConnected]);
        $path = $request->input('path');

        // Check is responsible for dispatching PageVisited event !
        PageVisited::dispatchIf($isConnected && ! empty($path), $path);

        // Make sure anti loop cookie is delete
        if (Package::has('inside-authentication-windows') && Cookie::has(md5('askedkerberostoken'))) {
            Log::info('[windows auth] Anti loop cookie has been set to be forgotten');

            return $response->withCookie(Cookie::forget(md5('askedkerberostoken')));
        }

        return $response;
    }

    /**
     * Get the current user authentication status
     *
     * @OA\Get(
     *      path="/authentication/check",
     *      operationId="authenticationCheck",
     *      tags={"Authentification"},
     *      summary="Vérifier si un utilisateur est connecté",
     *      @OA\Response(
     *          response=200,
     *          description="Le status de la connection",
     *          @OA\MediaType(
     *           mediaType="application/json",
     *           @OA\Schema(
     *            type="object",
     *           @OA\Property(
     *                     property="data",
     *                     type="boolean",
     *                     description="Le status de connection de l'utilisateur courant",
     *                  )
     *               )
     *           )
     *       )
     *  )
     */
}
