<?php

namespace Inside\Ai\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Ai\Contracts\AIProviderContract;
use Inside\Ai\Contracts\TextProcessorContract;
use Inside\Ai\DTO\TextProcessingResult;
use Inside\Ai\Exceptions\EmptyTextException;
use Inside\Ai\Exceptions\InvalidActionException;
use Inside\Authentication\Models\User;

class TextProcessor implements TextProcessorContract
{
    public function __construct(
        private array $availableActions,
        private AIProviderContract $aiProvider
    ) {

    }

    public function processText(string $text, string $action, ?User $user = null): TextProcessingResult
    {
        try {
            if (empty(trim($text))) {
                throw new EmptyTextException();
            }

            if (!$this->isValidAction($action)) {
                throw new InvalidActionException(Lang::get('validation.inalide_action',['action' => $action]));
            }

            $langcode = $user?->langcode ?? config('app.locale');
            $fullPrompt = $this->buildPrompt($action, $langcode);

            $cacheTag = md5($fullPrompt);
            $cacheKey = md5($text);

            $textProcessinOptions = [
                'action' => $action
            ];

            $cachedTextProcessingResult = Cache::tags([$cacheTag])->get($cacheKey);

            if ($cachedTextProcessingResult !== null) {
                return $cachedTextProcessingResult;
            }

            $aiResponse = $this->aiProvider->processText($text, $fullPrompt, $textProcessinOptions);

            if (!$aiResponse->success) {
                return TextProcessingResult::failure(
                    originalText: $text,
                    action: $action,
                    error: $aiResponse->error ?? Lang::get('validation.error_process'),
                );
            }

            $textProcessingResult = TextProcessingResult::success(
                originalText: $text,
                processedText: $aiResponse->processedText,
                action: $action,
                metadata: [
                    'processing_time' => microtime(true),
                    'ai_metadata' => $aiResponse->metadata
                ]
            );

            Cache::tags([$cacheTag])->put($cacheKey, $textProcessingResult, now()->addMinutes(15));

            return $textProcessingResult;

        } catch (EmptyTextException|InvalidActionException $exception) {
            return TextProcessingResult::failure(
                originalText: $text,
                action: $action,
                error: $exception->getMessage(),
            );
        } catch (\Exception $e) {
            Log::error('Text processing error', [
                'action' => $action,
                'user_id' => $user?->uuid,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return TextProcessingResult::failure(
                originalText: $text,
                action: $action,
                error: Lang::get('validation.unexpected_error'),
            );
        }
    }

    private function isValidAction(string $action): bool
    {
        return in_array($action, $this->availableActions);
    }

    private function buildPrompt(string $action, string $langocde): string
    {
        $actionPrompt = config("ai.prompts.actions.{$action}.{$langocde}", '');
        $originalPrompt = config("ai.prompts.original.{$langocde}", '');
        $formattingPrompt = config("ai.prompts.formatting.{$langocde}", '');
        $prompt = trim($originalPrompt . "\n\n" . $formattingPrompt . "\n\n" . $actionPrompt);

        return $prompt;
    }
}
