<?php

namespace Inside\Ai\Services\Providers;

use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Ai\Contracts\AIProviderContract;
use Inside\Ai\DTO\AIResponse;
use Inside\Ai\Exceptions\AIConfigurationException;
use Inside\Ai\Exceptions\AIProviderException;
use Inside\Facades\Http;

class OpenAIProvider implements AIProviderContract
{
    private const DEFAULT_MAX_TOKEN = 4096;
    private const DEFAULT_TEMPERATURE = 0.3;
    private const TEMPERATURES = [
        'correct' => 0.2,
        'rewrite' => 0.3,
        'formalize' => 0.4,
        'shortening' => 0.5,
        'analize' => 0.6
    ];

    public function __construct(
        private string $apiKey,
        private string $apiUrl,
        private array $models,
        private string $defaultModel
    )
    {}

    public function processText(string $text, string $prompt, array $options = []): AIResponse
    {
        try {
            $model = $options['model'] ?? $this->defaultModel;
            $temperature = $this->getTemperature($options);

            $messages = [
                ['role' => 'system', 'content' => $prompt],
                ['role' => 'user', 'content' => $text]
            ];

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->apiKey,
                'Content-Type' => 'application/json',
            ])->timeout(60)->post($this->apiUrl, [
                'model' => $model,
                'messages' => $messages,
                'max_tokens' => self::DEFAULT_MAX_TOKEN,
                'temperature' => $temperature,
            ]);

            if (!$response->successful()) {
                Log::error('OpenAI API Error', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                throw new AIProviderException(Lang::get('validation.error_call_api', ['body' => $response->body()]));
            }

            $data = $response->json();
            if (is_null(data_get($data, 'choices.0.message.content'))) {
                throw new AIProviderException(Lang::get('validation.invalide_openai_response'));
            }

            $processedText = trim(data_get($data, 'choices.0.message.content'));

            return AIResponse::success(
                processedText: $processedText,
                metadata: [
                    'model' => $model,
                    'prompt_tokens' => data_get($data, 'usage.prompt_tokens', 0),
                    'completion_tokens' => data_get($data, 'usage.completion_tokens', 0),
                ]
            );

        } catch (AIProviderException $e) {
            throw $e;
        } catch (\Exception $e) {
            Log::error('OpenAI Processing Error', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return AIResponse::failure(Lang::get('validation.openai_error_process', ['message' => $e->getMessage()]));

        }
    }

    private function getTemperature($option): float
    {
        return data_get(self::TEMPERATURES, $option['action'], self::DEFAULT_TEMPERATURE);
    }
}
