<?php

namespace Inside\User\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\Schema;
use Inside\Host\Bridge\BridgeContent;
use Inside\Jobs\Bus\Dispatchable;
use Inside\User\Events\UserDisabledEvent;
use Inside\User\Models\User;

class DisableUserJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;

    private const WHITELIST = [
        'name',
        'email',
        'password',
        'status',
        'langcode',
        'is_maintenance',
        'is_rgpd_compliant',
        'firstname',
        'lastname',
    ];

    /**
     * @var string
     */
    private $userUuid;

    /**
     * @param string $userUuid
     */
    public function __construct(string $userUuid)
    {
        $this->userUuid = $userUuid;
    }

    /**
     * @return void
     */
    public function handle()
    {
        /** @var User|null $user */
        $user = User::find($this->userUuid);
        if ($user === null || $user->information === null || $user->permission === null) {
            return;
        }

        // if decommissioning not activated
        if (! config('users.decommissioning', false)) {
            return;
        }

        if ($user->information->is_maintenance) {
            Log::warning(
                '[inside-user] Utilisateur <'.$user->uuid
                .'> ne peut être décommissionné car c\'est un compte de maintenance.'
            );

            return;
        }

        if ($user->permission->hasAnyRole(
            array_unique(array_merge(['super_administrator'], config('users.roles_whitelist')))
        )
        ) {
            Log::warning(
                '[inside-user] Utilisateur <'.$user->uuid
                .'> ne peut être décommissionné car il appartient à un role whitelisted.'
            );

            return;
        }
        Log::info('[inside-user] Utilisateur <'.$user->uuid.'> va être décommissionné.');
        $this->cleanupUserinfo($user);

        UserDisabledEvent::dispatch($user->uuid);
    }

    /**
     * @param User $user
     * @return void
     * @throws \Exception
     */
    protected function cleanupUserinfo(User $user)
    {
        $fieldNames = Schema::getFieldListing('users');
        $data = [];
        foreach ($fieldNames as $fieldName) {
            if (in_array($fieldName, self::WHITELIST)) {
                continue;
            }
            $data[$fieldName] = null;
        }
        $data['uuid'] = $user->uuid;
        Log::debug('[inside-user] Suppression des données utilisateurs avec => '.json_encode($data));
        $bridge = new BridgeContent();
        $bridge->contentUpdate('users', $data);
    }
}
