<?php

declare(strict_types=1);

namespace Inside\User\Http\Controllers;

use Exception;
use Illuminate\Auth\Events\PasswordReset;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Password;
use Illuminate\Validation\ValidationException;
use Inside\Authentication\Exceptions\AuthenticationException;
use Inside\Authentication\Models\User as AuthenticationUser;
use Inside\Content\Models\Contents\Users;
use Inside\Host\Bridge\BridgeContent;
use Inside\Search\Facades\Searchable;
use Inside\Settings\Models\Setting;
use Inside\User\Http\Controllers\Controller as BaseController;
use Inside\User\Models\User;
use Symfony\Component\HttpFoundation\StreamedResponse;

class UserController extends BaseController
{
    private const MAIL_NOTIFICATION_DISABLED_MESSAGE = 'validation.mail.notifications.disabled';

    public function avatar(Request $request): JsonResponse | StreamedResponse
    {
        $uuid = $request->route('id');
        $user = Users::find($uuid);

        if (is_null($user)) {
            return response()->json(
                [
                    'success' => false,
                    'message' => 'Not found',
                ],
                404
            );
        }

        $path = storage_path('app/'.$user->image);

        if (is_null($user->image) || ! File::exists($path)) {
            $avatar = file_get_contents(__DIR__.'/../../../resources/assets/img/avatar.png');

            return response()->stream(
                function () use ($avatar) {
                    echo $avatar;
                },
                200,
                [
                    'Content-Type' => 'image/png',
                    'Content-Disposition' => 'inline; filename="avatar.png"',
                ]
            );
        }

        $avatar = file_get_contents($path);
        $mimeType = (new \finfo(FILEINFO_MIME_TYPE))->buffer($avatar);

        return response()->stream(
            function () use ($avatar) {
                echo $avatar;
            },
            200,
            [
                'Content-Type' => $mimeType,
                'Content-Disposition' => 'inline; filename='.basename($user->image),
            ]
        );
    }

    public function update(Request $request): JsonResponse
    {
        /** @var array $route */
        $route = $request->route();
        $uuid = $route[2]['id'];

        // Check that the user changing information is the one logged in
        if (Auth::user() === null || Auth::user()->uuid != $uuid) {
            throw new AuthenticationException();
        }
        $user = Users::findOrFail($uuid);
        $bridge = new BridgeContent();
        $userModelIsSearchable = Searchable::isClassSearchable(Users::class);
        try {
            if ($userModelIsSearchable && $request->has('noindex')) {
                Users::disableSearchSyncing();
            }
            // on some instances, updating the user only once does not translate all pivots
            $bridge->contentUpdate('users', $request->merge(['uuid' => $user->uuid])->except('noindex'));
            $bridge->contentUpdate('users', $request->merge(['uuid' => $user->uuid])->except('noindex'));
            if ($userModelIsSearchable && $request->has('noindex')) {
                Users::enableSearchSyncing();
            }
            Log::debug(
                'The user has been updated',
                [
                    'name' => $user->name,
                    'uuid' => Auth::user()->uuid,
                    'type' => 'users',
                    'content uuid' => $uuid,
                ]
            );
        } catch (Exception $e) {
            if ($userModelIsSearchable && $request->has('noindex')) {
                Users::enableSearchSyncing();
            }
            Log::error($e->getMessage());
        }

        return response()->json(
            [
                'success' => true,
            ]
        );
    }

    public function sendResetLinkEmail(Request $request): JsonResponse
    {
        $isEmailSendingEnabled = Setting::where('key', 'email_sending_enabled')->first()?->value;

        if (! $isEmailSendingEnabled) {
            return $this->sendFailedResponse(self::MAIL_NOTIFICATION_DISABLED_MESSAGE);
        }

        $this->validateEmail($request);

        // Ask our password broker to send a reset link*
        $response = $this->broker()->sendResetLink($request->only('email'));

        // Send 200 response even if the user email does not exists
        if ($response == Password::INVALID_USER) {
            Log::info('[ResettingPassword] User not found !', $request->only('email'));

            return $this->sendResetLinkResponse($request, Password::RESET_LINK_SENT);
        }

        if (App::isProduction()) {
            return $this->sendResetLinkResponse($request, $response);
        }

        if ($response != Password::RESET_LINK_SENT) {
            $this->sendResetLinkFailedResponse($request, $response);
        }

        return $this->sendResetLinkResponse($request, $response);
    }

    public function sendCreateLinkEmail(Request $request): JsonResponse
    {
        $isEmailSendingEnabled = Setting::where('key', 'email_sending_enabled')->first()?->value;

        if (! $isEmailSendingEnabled) {
            return $this->sendFailedResponse(self::MAIL_NOTIFICATION_DISABLED_MESSAGE);
        }

        $this->validateEmail($request);

        // Ask our password broker to send a reset link*
        $response = Password::broker('users_welcome')->sendResetLink($request->only('email'), function ($user, $token) {
            $user = AuthenticationUser::findOrFail($user->uuid);
            $user->sendPasswordCreateNotification($token);
        });

        if (App::isProduction()) {
            return $this->sendResetLinkResponse($request, $response);
        }

        if ($response != Password::RESET_LINK_SENT) {
            $this->sendResetLinkFailedResponse($request, $response);
        }

        return $this->sendResetLinkResponse($request, $response);
    }

    protected function validateEmail(Request $request): void
    {
        $this->validate($request, ['email' => 'required|email']);
    }

    /**
     * Get our password broker
     *
     * Set in <project>-back/config/auth.php
     *
     * @return mixed
     */
    public function broker()
    {
        return Password::broker();
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function validateToken(Request $request): JsonResponse
    {
        $credentials = $request->only('email', 'token');

        if (is_null($user = $this->broker()->getUser($credentials)) || ! $this->broker()->tokenExists($user, $credentials['token'])) {
            return response()->json([
                'valid' => false,
            ], 406);
        }

        return response()->json([
            'valid' => true,
        ]);
    }

    public function reset(Request $request): JsonResponse
    {
        if ($user = Auth::user()) {
            $user = User::find($user->uuid);
            if ($user) {
                $token = $this->broker()->createToken($user);
                $request->merge(['token' => $token, 'email' => $user->email]);
            }
        }

        $this->validate($request, $this->rules(), $this->validationErrorMessages());

        // Ask the password broker to reset the password
        $response = $this->broker()->reset(
            $this->credentials($request),
            function ($user, $password) {
                $this->resetPassword($user, $password);
            }
        );

        if ($response != Password::PASSWORD_RESET) {
            $this->sendResetFailedResponse($request, $response);
        }

        return $this->sendResetResponse($response, $request->get('email'));
    }

    protected function rules(): array
    {
        return [
            'token' => 'required',
            'email' => 'required|email',
            // Each project may have its own password rule ( default to min 12 characters with uppercase and special symbol )
            'password' => config('app.password_rules', [
                'required',
                'confirmed',
                'min:12',
                'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[a-zA-Z])(?=.*[!@#*$%^&(){}[\]:;<>,.?~_+\-=\|]).+$/i',
            ]),
        ];
    }

    /**
     * @param string $key
     * @param string $rule
     * @return string
     */
    protected function getValidationTranslation(string $key, string $rule): string
    {
        $fullKey = 'validation.'.$key.'.'.$rule;
        $message = Lang::get($fullKey);
        if ($message != $fullKey) {
            return $message;
        }
        /** @var string $message */
        $message = trans($fullKey);
        if ($message != $fullKey) {
            return $message;
        }

        /** @var string $trans */
        $trans = __('validation.'.$rule);

        return $trans;
    }

    protected function validationErrorMessages(): array
    {
        return [
            'password.regex' => $this->getValidationTranslation('password', 'regex'),
            'password.required' => $this->getValidationTranslation('password', 'required'),
            'password.confirmed' => $this->getValidationTranslation('password,', 'confirmed'),
            'password.min' => $this->getValidationTranslation('password', 'min.string'),
            'token.required' => $this->getValidationTranslation('token', 'required'),
            'email.required' => $this->getValidationTranslation('email', 'required'),
            'email.email' => $this->getValidationTranslation('email', 'email'),
        ];
    }

    protected function credentials(Request $request): array
    {
        return $request->only('email', 'password', 'password_confirmation', 'token');
    }

    protected function resetPassword(AuthenticationUser $user, string $password): JsonResponse
    {
        $user->password = $password;
        $user->save();
        event(new PasswordReset($user));

        return response()->json([]);

        /**
         * NOTE: auto login is not used by front and meaningless
         * TODO: check security & front to see if we keep that in future
         *
         * // Login after reset password
         * Authentication::login(['email' => $user->email, 'password' => $password]);
         *
         * $transformed              = $user->toArray();
         * $transformed['api_token'] = $user->api_token;
         *
         * // Set header so we can use auth api guard
         * request()->headers->set('api_token', $user->api_token);
         *
         * return response()->json(['data' => $transformed]);
         */
    }

    /**
     * @param string $response
     * @param string $email
     * @return JsonResponse
     */
    protected function sendResetResponse(string $response, string $email): JsonResponse
    {
        /** @var User $user */
        $user = User::where('email', $email)->firstOrFail();
        $transformed = $user->toArray();
        $transformed['api_token'] = $user->api_token;

        return response()->json(
            [
                'success' => true,
                'message' => Lang::get($response),
                'body' => Lang::get($response),
                'data' => $transformed,
            ]
        );
    }

    protected function sendResetFailedResponse(Request $request, string $response): void
    {
        if ($request->wantsJson()) {
            throw ValidationException::withMessages(
                [
                    'email' => [Lang::get($response)],
                ]
            );
        }
    }

    protected function sendResetLinkResponse(Request $request, string $response): JsonResponse
    {
        return response()->json(['status' => 200, 'body' => Lang::get($response)]);
    }

    protected function sendResetLinkFailedResponse(Request $request, string $response): void
    {
        throw ValidationException::withMessages(
            [
                'email' => [Lang::get($response)],
            ]
        );
    }

    protected function sendFailedResponse(string $response): JsonResponse
    {
        return response()->json(['message' => Lang::get($response)], 400);
    }
}
