<?php

namespace Inside\Import\Services;

use Illuminate\Support\Facades\Log;
use Inside\Content\Services\Schema\SchemaService;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;

final class UserReferencesCleaner
{
    /**
     * @return void
     */
    public function cleanUnusedReferences(): void
    {
        $schemaService = new SchemaService();
        $userReferencesConfig = config('user_references');
        if (empty($userReferencesConfig['content_types'])) {
            return;
        }

        foreach ($userReferencesConfig['content_types'] as $contentType) {
            $parentContentTypes = $schemaService->getFieldNamesThatReferenceType($contentType);
            if (empty($parentContentTypes)) {
                continue;
            }
            $this->cleanUnusedReferencesForContent($contentType, array_keys($parentContentTypes));
        }
    }

    private function cleanUnusedReferencesForContent(string $contentType, array $parentContentTypes): void
    {
        $bridge = new BridgeContent();
        $reveserContentTypes = collect($parentContentTypes)
            ->map(fn ($type) => str($type)
                ->camel()
                ->ucfirst()
                ->prepend('reverse')
                ->toString()
            )
            ->toArray();

        $query = type_to_class($contentType)::with($reveserContentTypes)
            ->where('status', true);

        foreach ($reveserContentTypes as $reveserContentType) {
            $query->whereDoesntHave($reveserContentType, function (Builder $query) use ($reveserContentType) {
                $query->where('status', true)
                    ->when(
                        $reveserContentType === 'reverseUsers',
                        fn (Builder $query) => $query->where('is_maintenance', false)
                    );
            })->whereNotIn(
                'uuid_host',
                type_to_class($contentType)::with([$reveserContentType])
                    ->where('status', true)
                    ->whereHas($reveserContentType, function (Builder $query) use ($reveserContentType) {
                        $query->where('status', true)
                            ->when(
                                $reveserContentType === 'reverseUsers',
                                fn (Builder $query) => $query->where('is_maintenance', false)
                            );
                    })->get()->pluck('uuid_host')
            );
        }

        $contents = $query->get();

        $titles = $contents->pluck('title');
        $contents->each(fn ($content) => $bridge->contentUpdate($contentType, [
            'uuid' => $content->uuid,
            'status' => 0,
        ]));

        Log::channel('import')->info(
            sprintf('%d Models (%s) deactivated', $titles->count(), $contentType),
            [
                'contents' => $titles,
            ]
        );
    }
}
