<?php

namespace Inside\Import\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Inside\Facades\Package;
use Inside\Import\Events\RunUserImportCommand;
use Inside\Import\Models\UsersImport;
use InvalidArgumentException;
use Symfony\Component\Process\Process;

class UsersImportController extends Controller
{
    private const AZUREAD_PROVIDER = 'azuread';

    private const CSV_PROVIDER = 'csv';

    private const XLSX_PROVIDER = 'xlsx';

    private const LDAP_PROVIDER = 'ldap';

    private const GOOGLE_PROVIDER = 'google';

    private const RIGHT_PROVIDER = 'right';

    private const ONELOGIN_PROVIDER = 'onelogin';

    public function latest(): JsonResponse
    {
        return response()->json([
            'data' => UsersImport::latest()->first(),
        ]);
    }

    private function isDefaultConfiguration(string $provider): bool
    {
        if ($provider === self::RIGHT_PROVIDER) {
            return false;
        }

        $default = require cms_base_path("vendor/maecia/inside-user-$provider/config/$provider.php");
        $config = config($provider);

        return collect(json_encode($config))->diff(json_encode($default))->isEmpty();
    }

    private function isProviderConfigured(string $provider): bool
    {
        return match ($provider) {
            self::LDAP_PROVIDER => ! $this->isDefaultConfiguration('ldap'),
            self::AZUREAD_PROVIDER => ! $this->isDefaultConfiguration('azuread'),
            self::GOOGLE_PROVIDER => ! $this->isDefaultConfiguration('google'),
            self::ONELOGIN_PROVIDER => ! $this->isDefaultConfiguration('onelogin'),
            self::RIGHT_PROVIDER => ! $this->isDefaultConfiguration('right'),
            self::CSV_PROVIDER => ! empty(config('csv')),
            self::XLSX_PROVIDER => ! empty(config('xlsx')),
            default => false
        };
    }

    public function getProviders(): JsonResponse
    {
        $types = array_keys(config('import.types', []));

        $providers = collect($types)->flatMap(function ($providerType) {
            if (! $this->isProviderConfigured($providerType)) {
                return null;
            }

            $importerClass = config("import.types.$providerType.class");
            $importer = new $importerClass();

            if (config($providerType) !== null) {
                return [
                    $providerType => [
                        'providerType' => array_keys($importer->getEntries()),
                        'providerIdentifier' => $this->getProviderIdentifier($providerType),
                    ],
                ];
            }
        })->filter()->toArray();

        return response()->json([
            'data' => $providers,
        ]);
    }

    public function getUserFromAD(Request $request, string $provider): JsonResponse
    {
        $defaultPackage = $provider === self::XLSX_PROVIDER ? 'inside-user-csv' : 'inside-user-'.$provider;
        if (
            ! Package::has($defaultPackage) &&
            ! Package::has('lhg-commons')
        ) {
            throw new InvalidArgumentException(
                "Le module {$provider} n'est pas un module maecia valide ou n'est pas installé sur cette instance."
            );
        }

        try {
            $process = Process::fromShellCommandline('php artisan '.$this->getArtisanCommand($provider, $request));

            $process->run();
        } catch (Exception $e) {
            throw new Exception("$provider is not correctly configured", $e->getCode());
        }

        return response()->json([
            'data' => [
                'code' => $process->getExitCode(),
                'value' => $process->getOutput(),
            ],
        ]);
    }

    public function runUsersImport(Request $request, string $commandName): JsonResponse
    {
        $userImportScheduleCommands = $this->getUserImportScheduleCommands();
        if (isset($userImportScheduleCommands[$commandName])) {
            try {
                $user = $request->user();
                RunUserImportCommand::dispatch($userImportScheduleCommands[$commandName], $user);
            } catch (Exception $e) {
                return response()->json([
                    'data' => [
                        'message' => $e->getMessage(),
                    ],
                ], 502);
            }

            return response()->json([
                'data' => [
                    'message' => 'Import started successfully.',
                ],
            ]);
        }

        return response()->json([
            'data' => [
                'message' => 'Unable to find a scheduled command matching the provided name !',
            ],
        ], 404);
    }

    public function listUserImportScheduleCommands(): JsonResponse
    {
        return response()->json(
            $this->getUserImportScheduleCommands()
        );
    }

    private function getUserImportScheduleCommands(): array
    {
        $process = Process::fromShellCommandline('php artisan schedule:list --no-ansi --no-interaction');
        $process->run();

        if (! $process->isSuccessful()) {
            throw new Exception('Unable to retrieve scheduled commands: '.$process->getErrorOutput());
        }

        $schedule = $process->getOutput();

        return str($schedule)
            ->explode("\n")
            ->filter(fn ($event) => str_contains($event, 'inside:user:import'))
            ->mapWithKeys(function ($event) {
                $command = str($event)->after('php artisan')->before('Next Due')->trim()->trim('.');
                $provider = $command->after('inside:user:import')->before('-')->trim();

                return [$provider->toString() => $command->toString()];
            })
            ->toArray();
    }

    private function getArtisanCommand(string $provider, Request $request): string
    {
        $identifier = $request->get('identifier');
        $entry = $request->get('entry', 'default');
        $ignorePostfilter = false;
        $ignoreGroups = false;

        $command = "inside:user:dump {$provider} {$identifier} --entry={$entry}";

        if ($provider === self::AZUREAD_PROVIDER) {
            $ignorePostfilter = $request->get('ignorePostfilter', 'false') !== 'false';
            $ignoreGroups = $request->get('ignoreGroups', 'false') !== 'false';
        }

        if ($ignorePostfilter) {
            $command .= ' --disable-postfilter';
        }
        if ($ignoreGroups) {
            $command .= ' --disable-group-filter';
        }

        return $command;
    }

    private function getProviderIdentifier(string $providerType): string
    {
        $result = match ($providerType) {
            self::LDAP_PROVIDER => config('ldap_auth.identifiers.ldap.locate_users_by'),
            self::AZUREAD_PROVIDER => config("$providerType.sync_attributes.mail") ?? config("$providerType.sync_attributes.email"),
            self::GOOGLE_PROVIDER => config("$providerType.sync_attributes.mail") ?? config("$providerType.sync_attributes.email"),
            self::CSV_PROVIDER => collect(config("$providerType.files"))->pluck('fields.email')->first() ? 'email' : 'mail',
            self::XLSX_PROVIDER => collect(config("$providerType.files"))->pluck('fields.email')->first() ? 'email' : 'mail',
            self::ONELOGIN_PROVIDER => config("$providerType.sync_attributes.email"),
            self::RIGHT_PROVIDER => 'id',
            default => '',
        };

        return is_callable($result) ? 'Dynamique' : $result;
    }
}
