<?php

namespace Inside\Import\Console;

use Illuminate\Console\Command;
use Illuminate\Contracts\Filesystem\FileNotFoundException;
use Illuminate\Filesystem\Filesystem;

class SyncroMakeCommand extends Command
{
    protected $name = 'inside:make:syncro';

    protected $signature = 'inside:make:syncro { type? : Type of import to be execute (if empty, display allowed types) }';

    protected $description = 'Generation of the import configuration';

    public function __construct(
        protected Filesystem $files
    ) {
        parent::__construct();
    }

    /**
     * @throws FileNotFoundException
     */
    public function handle(): void
    {
        /** @var string $providerType */
        $providerType = $this->argument('type');
        $this->displayConfiguredTypes($providerType);
        $nameProviderFile = $providerType === 'ldap' ? 'adldap_auth' : $providerType;
        $configPath = back_path('config/'.$nameProviderFile.'.php');

        if ($this->files->exists($configPath)) {
            $this->error('File already exists!');
            exit(1);
        }

        $this->files->put($configPath, $this->buildConfig($providerType));
        $this->info('File created successfully.');
    }

    /**
     * @param string|null $providerType
     * @return void
     */
    protected function displayConfiguredTypes(?string $providerType): void
    {
        $types = array_keys(config('import.types', []));
        if (empty($types)) {
            $this->error('There is no import types configured');
        } elseif (! in_array($providerType, $types)) {
            $this->info('Configured import types:');
            foreach ($types as $type) {
                $this->info(" - $type");
            }
            exit(1);
        }
    }

    /**
     * @throws FileNotFoundException
     */
    protected function buildConfig(string $providerType): string
    {
        $stub = $this->files->get($this->getStub($providerType));

        return $this->replaceProject($stub);
    }

    protected function replaceProject(string $stub): string
    {
        return str_replace('{{ project }}', env('APP_CODE'), $stub);
    }

    protected function getStub(string $providerType): string
    {
        return __DIR__.'/stubs/'.$providerType.'.stub';
    }
}
