<?php

namespace Inside\Import\Console;

use Closure;
use Illuminate\Console\Command;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Content\Models\Contents\Users;
use Inside\Database\Eloquent\Builder;
use Inside\Host\Bridge\BridgeContent;
use Inside\Import\Contracts\ImporterInterface;
use Inside\Import\Events\AfterUserImport;
use Inside\Import\Events\BeforeUserImport;
use Inside\Import\Events\ImportFinishedEvent;
use Inside\Import\Exceptions\InvalidReferenceException;
use Inside\Import\Facades\UserImport;
use Inside\Import\Models\UsersImport as UsersImportModel;
use Inside\Import\Services\UserReferencesCleaner;
use Inside\Import\Services\UserReferencesCleanerBis;
use Inside\Permission\Models\Role;
use Inside\Validation\Rule;
use Inside\Validation\ValidateRequests;
use League\Csv\CannotInsertRecord;
use League\Csv\Writer;

class ImportCommand extends Command
{
    use ValidateRequests;

    /**
     * @var string
     */
    protected $name = 'inside:user:import';

    /*
 * @var string
 */
    protected $signature = 'inside:user:import { type? : Type of import to be execute (if empty, display allowed types) }
                            { --S|silent }
                            { --e|entry=* }
                            { --delete : Delete all the users found by the import }
                            { --disable : Disable all the users found by the import }
                            { --start-from= }
                            { --disable-indexation }
                            { --F|filter= }
                            { --max= : maximum of users to import FOR ONE ENTRY }
                            { --limit= : limit of users to query from AD }
                            { --save-failed }
                            { --D|disable-not-imported : Disable all existing users not present in the import. (except maintenance and super_administrator) }
                            { --disable-old-entries : Disable all existing users on old or other entries }
                            { --test= : Path of test file }
                            { --clean-obsolete-providers : Disable all users from other providers }
                            { --notify-user= : Notify specific user }
                            { --process-tracker= : Process tracker to update progress bar }
                            ';

    /**
     * @var string
     */
    protected $description = 'User import';

    protected ImporterInterface $importer;

    public function __construct(private UserReferencesCleaner $userReferencesCleaner)
    {
        parent::__construct();
    }

    /**
     * Import users
     */
    public function handle(): void
    {
        $bar = null;

        /** @var array $entriesOption */
        $entriesOption = $this->option('entry');

        /** @var string|null $providerType */
        $providerType = $this->argument('type');

        /** @var string|null $filter */
        $filter = $this->option('filter');

        /** @var string|null $startFrom */
        $startFrom = $this->option('start-from');

        /** @var string|null $test */
        $test = $this->option('test');

        $delete = (bool) $this->option('delete');
        $disable = (bool) $this->option('disable');
        $silent = (bool) $this->option('silent');
        $disableIndexation = (bool) $this->option('disable-indexation');
        $saveFailed = (bool) $this->option('save-failed');
        $disableNotImported = (bool) $this->option('disable-not-imported');
        $disableOldEntries = (bool) $this->option('disable-old-entries');
        $cleanObsoleteProviders = (bool) $this->option('clean-obsolete-providers');
        $processTracker = (string) $this->option('process-tracker'); // @phpstan-ignore-line

        $max = (int) $this->option('max');
        $limit = (int) $this->option('limit');

        $importer = $this->getImporter($providerType, $silent);

        if ($providerType === null || $importer === null) {
            return;
        }

        $this->importer = $importer;

        Log::channel('import')->info('[ImportCommand] Début d\'importation avec ('.get_class($this->importer).')');

        if ($test) {
            $silent or $this->info('Mode TEST');
            if (! file_exists($test)) {
                $this->error("Test file [{$test}] does not exist");

                return;
            }
            $test = require_once $test;
        }

        $bridge = new BridgeContent();

        $preFlight = $this->importer->getPreFlight();
        if ($preFlight && is_callable($preFlight)) {
            Log::channel('import')->info('[ImportCommand] Preflight');
            $preFlight(($silent ? null : $this), $bridge);
        }

        $filter = $filter ?? $this->importer->getFilter();

        if ($filter) {
            $silent or $this->info('Using filter ['.$filter.']');
            Log::channel('import')->info('[ImportCommand] Filtre ('.$filter.')');
        }

        if ($disableIndexation) {
            Users::disableSearchSyncing();
        }

        $entries = $this->getEntries($entriesOption);

        if (empty($entries)) {
            $this->error('No entry found.');

            return;
        }
        Log::channel('import')->info('[ImportCommand] Importer nous a donné ('.count($entries).') provider.s');

        $totalImported = 0;
        $totalDisabled = 0;
        $totalFound = 0;
        $imported = [];
        foreach ($entries as $providerName => $entry) {
            $failed = null;
            if ($saveFailed) {
                $outputFile = cms_base_path(
                    env('APP_LOG_PATH', 'storage/logs').DIRECTORY_SEPARATOR."import_{$providerType}_".date(
                        'Y_m_d_His'
                    )."_$providerName.csv"
                );
                $failed = Writer::createFromPath($outputFile, 'w+');
            }

            // For retroactivity
            $enableRetroactivity = UserImport::rebuildReferencesIsNeeded($providerType, $providerName, $entry);

            try {
                $users = $this->importer->getUsers($entry, $filter, $test, $limit);
            } catch (\Exception $exception) {
                $this->error("Provider [$providerName] : ".$exception->getMessage());
                continue;
            }

            $globalPostFilter = config('import.postfilter');
            $typePostFilter = config("$providerType.postfilter");
            if ($globalPostFilter && ! $typePostFilter && is_callable($globalPostFilter)) {
                $usersCollection = collect($users);
                $users = $globalPostFilter($usersCollection, $providerName)->toArray();
            }

            $totalFound += count($users);

            if (! $silent) {
                $this->info('Processing entry "'.$providerName.'" ['.count($users).'] users found');
                if ($max) {
                    $this->info("Maximum users is set to [$max]");
                }
                $bar = $this->getOutput()->createProgressBar(! $max ? count($users) : $max);
                $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");
                if ($delete) {
                    $this->info('Starting to delete users from inside ...');
                } else {
                    $this->info('Starting to add users to inside ...');
                }
            }
            Log::channel('import')->info('[ImportCommand] Importation de ('.count($users).') résultat.s');

            $i = 0;
            $indexUser = 1;
            foreach ($users as $user) {
                if (! empty($processTracker) && $totalFound > 1) {
                    $this->updateProgressBar($indexUser, $totalFound, $processTracker);
                    $indexUser++;
                }

                if ($startFrom && $i <= ((int) $startFrom)) {
                    $i++;
                    continue;
                }

                if ($max && $i++ >= $max) {
                    break;
                }

                $data = $this->importer->getSyncAttributes($user, $entry, $bridge);

                // Fix weird old mail attribute system ( there is no mail column on users in inside system ... )
                if (isset($data['mail'])) {
                    $data['email'] = $data['mail'];
                    unset($data['mail']);
                }
                Log::channel('import')->info(str_repeat('#', 80));
                Log::channel('import')->info('[ImportCommand] Importation de ('.json_encode($data).')');

                if ($enableRetroactivity) {
                    $lumenUser = User::where('email', $data['email'])->first();

                    if (! $lumenUser) {
                        $lumenUser = User::where('name', $data['name'])->first();
                    }

                    if ($lumenUser) {
                        if (
                            ! UserImport::setUserReference(
                                $providerType,
                                $providerName,
                                $entry,
                                $data,
                                $lumenUser->uuid
                            )
                        ) {
                            Log::channel('import')->error(
                                '[ImportCommand] {retroactivité} Impossible de créer la référence utilisateur'
                            );
                            $bar and $bar->advance();
                            continue;
                        }
                        Log::channel('import')->info('[ImportCommand] {retroactivité} Référence correctement créée');
                    }
                }

                // Get Inside user from data reference
                if (isset($entry['discovering_attribute']) && $entry['discovering_attribute'] != 'email') {
                    try {
                        $lumenUser = UserImport::getUserFromEntry($providerType, $providerName, $entry, $data);
                    } catch (InvalidReferenceException $e) {
                        Log::channel('import')->error(
                            '[ImportCommand] InvalidReferenceException ('.$e->getMessage().')'
                        );
                        $bar and $bar->advance();
                        continue;
                    }
                    // fallback
                    if ($lumenUser === null) {
                        $lumenUser = User::where('email', $data['email'])->orWhere('name', $data['name'])->first();
                        if ($lumenUser) {
                            if (
                                ! UserImport::setUserReference(
                                    $providerType,
                                    $providerName,
                                    $entry,
                                    $data,
                                    $lumenUser->uuid
                                )
                            ) {
                                Log::channel('import')->error(
                                    '[ImportCommand] Impossible de créer la référence utilisateur'
                                );
                            }
                            Log::channel('import')->info('[ImportCommand] Référence correctement créée');
                        }
                    }
                } else {
                    $lumenUser = User::where('email', $data['email'])->first();
                }

                $newUser = false;
                if (! $lumenUser) {
                    $newUser = true;
                    $data['password'] = bcrypt(Str::random(32)); // if it is new user we put random password
                } else {
                    $data['uuid'] = $lumenUser->uuid;
                }

                $cleanData = $this->importer->getCleanData();
                // Save raw data
                $rawData = $data;

                if (is_object($cleanData) && $cleanData instanceof Closure) {
                    $cleanData($data);
                    Log::channel('import')->info('[ImportCommand] Data nettoyée ('.json_encode($data).')');
                } else {
                    // Use main content validation
                    try {
                        $data = ContentHelper::addMissings('users', $data);
                        $data = ContentHelper::castAttributes('users', $data);
                        $rules = ContentHelper::makeRules('users', ($newUser ? null : $lumenUser->uuid), true);

                        if (isset($entry['dont_check_email']) && $entry['dont_check_email']) {
                            foreach ($rules as $rk => $rule) {
                                foreach ($rule as $rkk => $r) {
                                    if (is_string($r) && Str::startsWith($r, 'emailpp')) {
                                        $rules[$rk][$rkk] = 'email';
                                    }
                                }
                            }
                        }
                        $data = $this->validateData($data, $rules);
                    } catch (ValidationException $e) {
                        Log::channel('import')->error(
                            '[ImportCommand] Validation échouée ('.collect($e->errors())->flatten().')'
                        );
                        $bar and $bar->advance();
                        continue;
                    }
                    Log::channel('import')->info('[ImportCommand] Data nettoyée ('.json_encode($data).')');
                }

                $bar and $bar->setMessage(' => '.$data['name'].' ['.$data['email'].']');

                if (! isset($data['status'])) {
                    $data['status'] = 1;
                }

                if ($disable) {
                    $data['status'] = 0;
                }

                BeforeUserImport::dispatch($data);

                // Check email and name ( should be ok, main validator already check that )
                if (($errors = $this->validateEmailAndName($data, $lumenUser)) !== true) {
                    Log::channel('import')->error(
                        '[ImportCommand] Validation email/name échouée ('.collect($errors)->flatten().')'
                    );
                    if (! $silent) {
                        $this->line('');
                        $this->error('Failed to import user ['.json_encode($data).']');
                        if ($saveFailed && $failed instanceof Writer) {
                            try {
                                $failed->insertOne($this->formatData($data));
                            } catch (CannotInsertRecord $e) {
                            }
                        }
                    }
                    $bar and $bar->advance();
                    continue;
                }

                if (
                    isset($entry['discovering_attribute'])
                    && ! InsideSchema::hasField(
                        'users',
                        $entry['discovering_attribute']
                    )
                    && isset($data[$entry['discovering_attribute']])
                ) {
                    unset($data[$entry['discovering_attribute']]);
                }
                Log::channel('import')->info(
                    '[ImportCommand] Data pour la création/modification ('.json_encode($data).')'
                );

                if ($delete && $lumenUser) {
                    Log::channel('import')->info('[ImportCommand] Suppression de ('.$lumenUser->email.')');
                    $bridge->contentDelete('users', $lumenUser->uuid);
                    $imported[] = $lumenUser->uuid;
                } else {
                    $uuid = null;
                    if ($newUser) {
                        Log::channel('import')->info('[ImportCommand] Ajout de l\'utilisateur');
                        try {
                            $uuid = $bridge->contentInsert('users', $data);
                            if (($uuid) === null) {
                                Log::channel('import')->error(
                                    '[ImportCommand] Echec de création du compte ['.$data['name'].']'
                                );
                                $this->error('Failed to Create / Insert Drupal User  ['.$data['name'].']');
                                continue;
                            }
                        } catch (\Throwable $e) {
                            Log::channel('import')->error(
                                '[ImportCommand] Echec de création du compte ['.$data['name'].'] => '
                                .$e->getMessage()
                            );
                            $this->error(
                                'Failed to Create / Insert Drupal User  ['.$data['name'].'] => '.$e->getMessage()
                            );
                        }
                        try {
                            if (! UserImport::setUserReference($providerType, $providerName, $entry, $rawData, $uuid)) {
                                Log::channel('import')->error(
                                    '[ImportCommand]  Impossible de créer la référence utilisateur'
                                );
                                // Duplicate
                                /** @var string $uuid */
                                $bridge->contentDelete('users', $uuid);
                                continue;
                            }
                        } catch (InvalidReferenceException $e) {
                            Log::channel('import')->error(
                                '[ImportCommand]  Impossible de créer la référence utilisateur'
                            );
                            // Duplicate
                            /** @var string $uuid */
                            $bridge->contentDelete('users', $uuid);
                            continue;
                        }
                    } else {
                        Log::channel('import')->info(
                            '[ImportCommand] Mise-à-jour de l\'utilisateur <'.$data['uuid'].'>'
                        );
                        try {
                            // Clear previous image before uploading a new one
                            if (! empty($data['image']) && ! empty($lumenUser->image)) {
                                $newImagePath = $data['image'];
                                $oldImagePath = $lumenUser->image;

                                $disk = Storage::disk('local');
                                if ($disk->exists($oldImagePath) && $disk->exists($newImagePath)) {
                                    $newHash = md5($disk->get($newImagePath));
                                    $oldHash = md5($disk->get($oldImagePath));

                                    if ($newHash === $oldHash) {
                                        unset($data['image']);
                                    } else {
                                        Storage::delete($oldImagePath);
                                    }
                                }
                            }

                            if (($uuid = $bridge->contentUpdate('users', $data)) === null) {
                                Log::channel('import')->error(
                                    '[ImportCommand] Echec de mise à jour du compte ['.$data['name'].']'
                                );
                                $this->error('Failed to Create / Insert Drupal User  ['.$data['name'].']');
                                continue;
                            }
                        } catch (\Throwable $e) {
                            Log::channel('import')->error(
                                '[ImportCommand] Echec de mise à jour du compte ['.$data['name'].'] => '
                                .$e->getMessage()
                            );
                            $this->error(
                                'Failed to Create / Insert Drupal User  ['.$data['name'].'] => '.$e->getMessage()
                            );
                            continue;
                        }
                    }
                    AfterUserImport::dispatch(Users::find($uuid), $lumenUser !== null);
                    $lumenUser = User::find($uuid);
                    $lumenUser->provider_name = $providerName;
                    $lumenUser->provider_type = $providerType;
                    $lumenUser->save();
                    $imported[] = $uuid;
                }
                $bar and $bar->advance();
            }

            if (! $silent && $bar) {
                $this->info('');
                $bar->setMessage(' => Done.');
                $bar->finish();
                $this->info('');
                $this->info(
                    'Done, '.($max ?: count($imported)).' users '.($delete ? 'deleted' : 'imported')
                    .' from entry '.$providerName.'.'
                );
                Log::channel('import')->info('[ImportCommand] Importation terminée');
                if ($disable) {
                    $this->info('Those users are disabled.');
                }
            }

            if ($disableNotImported && ! $test) {
                if (! empty($imported)) {
                    $totalDisabled += $this->disableUsers(
                        $bridge,
                        $silent,
                        $imported,
                        $providerType,
                        $providerName,
                        $entry['user_filter'] ?? null
                    );
                    if (in_array($providerType, ['csv', 'xlsx']) && ! empty(config('xlsx'))) {
                        $totalDisabled += $this->disableUsers(
                            $bridge,
                            $silent,
                            $imported,
                            $providerType === 'csv' ? 'xlsx' : 'csv',
                            $providerName,
                            $entry['user_filter'] ?? null
                        );
                    }
                    $this->info("Total users disabled: [$totalDisabled] ");
                } else {
                    if (! $silent) {
                        Log::channel('import')->info("[ImportCommand] Aucun utilisateur importé pour le provider '$providerName', on ne désactive pas les utilisateurs.");
                        $this->info('No user imported, we do not disable any user');
                    }
                }
            }

            $totalImported += count($imported);
        }

        // disable old entries
        if ($disableOldEntries && $disableNotImported && ! $test && $totalImported) {
            $existingEntries = array_keys($entries);
            $oldEntries =
        User::whereNotIn('provider_name', $existingEntries)->pluck('provider_name')
          ->unique();
            foreach ($oldEntries as $entry) {
                $totalDisabled += $this->disableUsers($bridge, $silent, [], $providerType, $entry);
            }
        }

        if ($cleanObsoleteProviders && ! $test) {
            $obsoleteProviders = User::where('provider_type', '!=', $providerType)
        ->pluck('provider_type')
        ->unique();

            foreach ($obsoleteProviders as $provider) {
                $oldEntries = User::where('provider_type', $provider)
          ->pluck('provider_name')
          ->unique();

                foreach ($oldEntries as $entry) {
                    $totalDisabled += $this->disableUsers($bridge, $silent, [], $provider, $entry);
                }
            }
        }

        $postFlight = $this->importer->getPostFlight();
        if (is_object($postFlight) && $postFlight instanceof Closure) {
            $postFlight(($silent ? null : $this), $imported, $bridge);
        }

        $this->userReferencesCleaner->cleanUnusedReferences();

        $silent or $this->info("Total users imported: [$totalImported].");

        UsersImportModel::create([
            'provider_type' => $providerType,
            'users_found' => $totalFound,
            'users_imported' => $totalImported,
            'users_disabled' => $totalDisabled,
        ]);

        $notifyUserUuid = $this->option('notify-user');
        if ($notifyUserUuid) {
            $notifyUser = User::findOrFail($notifyUserUuid);
            event(new ImportFinishedEvent(
                $totalFound,
                $totalImported,
                $totalDisabled,
                $notifyUser
            ));
        }
    }

    /**
     * get Importer
     *
     * @param string|null $providerType
     * @param bool $silent
     * @return ImporterInterface|null
     */
    protected function getImporter(?string $providerType, bool $silent = false): ?ImporterInterface
    {
        if ($providerType === null) {
            $this->displayConfiguredTypes();

            return null;
        }

        $importerClass = config("import.types.$providerType.class"); // types have to be declared in the import.php file

        if (! $importerClass) {
            $silent or $this->error("The import type '$providerType' is not defined.");
            exit(-1);
        }

        $importers = class_implements($importerClass);
        if ($importers && ! in_array(ImporterInterface::class, $importers)) {
            $silent or $this->error(
                'The class '.$importerClass.' doest not implement the interface '.ImporterInterface::class
            );
            exit(-1);
        }

        /** @var ImporterInterface $importer */
        $importer = new $importerClass();

        if (! $importer->prepareImport()) {
            $silent or $this->error('Wrong configurations for importer "'.$providerType.'"');
            exit(-1);
        }

        return $importer;
    }

    /**
     * Get entries
     *
     * @param array $entriesOption
     * @return array
     */
    protected function getEntries(array $entriesOption): array
    {
        $entries = $this->importer->getEntries();

        if (! empty($entriesOption)) {
            $entries = Arr::only($entries, $entriesOption);
            $notFound = array_diff($entriesOption, array_keys($entries));
            if (! empty($notFound)) {
                $this->error('Entries ['.implode(',', $notFound).'] not found.');
            }
        }

        return $entries;
    }

    /**
     * Disable users
     *
     * @param BridgeContent $bridge
     * @param bool $silent
     * @param array $whiteList
     * @param string $providerType
     * @param string $providerName
     * @param string|null $filter
     * @throws \Exception
     */
    protected function disableUsers(
    BridgeContent $bridge,
    bool $silent,
    array $whiteList,
    string $providerType,
    string $providerName,
    ?string $filter = null
  ): int {
        $bar = null;
        $table = type_to_table('users');
        $hasFilter = $filter && $table && Schema::hasColumn($table, $filter);
        $superAdminRole = Role::query()->where('name', 'super_administrator')->first();
        $superAdmins = $superAdminRole instanceof Role ? $superAdminRole->users->pluck('uuid') : [];

        // on supprime tous les users qui ne sont ni maintenance, ni super admin
        $query = Users::query()
      ->join('inside_users', 'inside_users.uuid', '=', "$table.uuid")
      ->where('provider_name', $providerName)
      ->where('provider_type', $providerType)
      ->where("$table.status", 1)
      ->where('is_maintenance', false)
      ->whereNotIn("$table.uuid", $superAdmins)
      ->when($hasFilter, fn (Builder $query) => $query->where($filter, '=', 1));

        // Laravel does not handle well WHERE NOT IN queries with 2k+ entries
        // Maybe we should fix it somewhere else ?
        foreach (array_chunk($whiteList, 500) as $chunk) {
            $query->whereNotIn("$table.uuid", $chunk);
        }

        $users = $query->get();
        if ($users->isEmpty()) {
            $silent or $this->info('No user to disable');

            return 0;
        }
        if (! $silent) {
            Log::channel('import')->info('[ImportCommand] Désactivation de ['.count($users).'] utilisateur.s');
            $this->info('Disabling activated users ['.count($users).']...');
            $bar = $this->getOutput()->createProgressBar(count($users));
            $bar->setFormat("%message%\n %current%/%max% [%bar%] %percent:3s%%");
        }
        foreach ($users as $user) {
            $bridge->contentUpdate('users', ['uuid' => $user->uuid, 'status' => 0]);
            if (! $silent) {
                $bar->setMessage(' => '.$user->name.' ['.$user->email.']');
                $bar->advance();
            }
        }
        if (! $silent) {
            $bar->setMessage(' => Done.');
            $bar->finish();
            Log::channel('import')->info('[ImportCommand] Désacactivation terminée');
            $this->info('');
            $this->info('All users are disabled.');
            $this->info('');
        }

        return $users->count();
    }

    /**
     * format Data
     *
     * @param array $data
     * @return array
     */
    protected function formatData(array $data): array
    {
        $formatted = [];
        foreach ($data as $key => $value) {
            if (is_array($value)) {
                $value = '['.implode(',', $value).']';
            }
            $formatted[] = "$key=>$value";
        }

        return $formatted;
    }

    /**
     * display configured type
     */
    protected function displayConfiguredTypes(): void
    {
        $types = array_keys(config('import.types', []));
        if (empty($types)) {
            $this->error('There is no import types configured');
        } else {
            $this->info('Configured import types:');
            foreach ($types as $type) {
                $this->info(" - $type");
            }
        }
    }

    /**
     * validate Email and Name
     *
     * @param array $data
     * @param User|null $user
     * @return array|bool
     */
    protected function validateEmailAndName(array $data, ?User $user = null)
    {
        $validator = Validator::make(
            $data,
            [
                'name' => [
                    'required',
                    $user !== null
                      ? Rule::unique('inside_users', 'name')->ignore($user->uuid, 'uuid')
                      : Rule::unique('inside_users', 'name'),
                    'max:255',
                ],
                'email' => [
                    'required',
                    'email',
                    $user !== null
                      ? Rule::unique('inside_users', 'email')->ignore($user->uuid, 'uuid')
                      : Rule::unique('inside_users', 'email'),
                    'max:255',
                ],
            ]
        );
        if ($validator->fails()) {
            return $validator->errors()->messages();
        }

        return true;
    }

    private function updateProgressBar(int $indexUser, int $totalFound, string $processTracker): void
    {
        $start = 20;
        $end = 80;
        $steps = $end - $start;

        if ($totalFound <= $steps) {
            $step = intdiv($steps, $totalFound - 1);
            $progress = $start + ($indexUser * $step);
        } else {
            $step = $totalFound / $steps;
            $progress = $start + intdiv($indexUser, (int) $step);
        }

        $progress = min((int) round($progress), $end);

        Cache::tags([$processTracker])->put('state', min(80, $progress), now()->addHours(2));
    }
}
