<?php

namespace Inside\Statistics\Services;

use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Comments;
use Inside\Notify\Models\NotificationLog;
use Inside\Permission\Facades\Role;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Repositories\StatisticsBetaRepository;

class AdvancedStatisticsService
{
    private string $firstDay;

    private string $lastDay;

    public function __construct(private StatisticsBetaRepository $statisticsRepository)
    {
        $this->firstDay = now()->startOfMonth()->toDateString();
        $this->lastDay = now()->endOfMonth()->toDateString();
    }

    public function getAttendenceStats(array $filters = []): array
    {
        $authenticatedUsers = $this->statisticsRepository
            ->getAuthenticationStatsQuery([
                'never_authenticated' => false,
                'from' => $this->firstDay,
                'to' => $this->lastDay,
            ])->count();

        $accessUsers = $this->statisticsRepository
            ->getAuthenticationStatsQuery([
                'never_access' => false,
                'from' => $this->firstDay,
                'to' => $this->lastDay,
            ])->count();

        $usersWithAuthenticatedRole = User::query()
            ->whereHas('roles', fn ($query) => $query->where('role_id', 9))
            ->count();

        $authenticatedPercentage = $usersWithAuthenticatedRole > 0
            ? round(($authenticatedUsers / $usersWithAuthenticatedRole) * 100, 2)
            : 0.00;

        $accessPercentage = $usersWithAuthenticatedRole > 0
            ? round(($accessUsers / $usersWithAuthenticatedRole) * 100, 2)
            : 0.00;

        return [
            'authenticated_nbr' => $authenticatedUsers,
            'access_nbr' => $accessUsers,
            'authenticated_percentage' => $authenticatedPercentage,
            'access_percentage' => $accessPercentage,
            'users_has_role_authenticated' => User::where('status', 1)->whereHas('roles', fn ($query) => $query->where('name', 'authenticated'))->count(),
        ];
    }

    public function getAttendenceStatsDisplay(int $monthId): array
    {
        $attendanceRow = DB::table('inside_advanced_stats_attendance')
            ->where('month_id', $monthId)
            ->first();

        return $attendanceRow ? $this->formatAttendance($attendanceRow) : [];
    }

    public function getActivityRoleBasedStats(array $filters = []): array
    {
        $users = User::with('roles')
            ->where('status', 1)
            ->get()
            ->flatMap(fn ($user) => $user->roles->pluck('name'))
            ->countBy();

        $usersInit = User::with('roles')
            ->where('status', 1)
            ->whereBetween('last_access_at', [
                $this->firstDay,
                $this->lastDay,
            ])
            ->get()
            ->flatMap(fn ($user) => $user->roles->pluck('name'))
            ->countBy();

        return $users->map(function ($totalCount, $role) use ($usersInit) {
            $activeCount = $usersInit->get($role, 0);
            $percentage = $totalCount > 0
                ? round(($activeCount / $totalCount) * 100, 2)
                : 0;

            return [
                'count' => $totalCount,
                'percentage' => $percentage,
            ];
        })->toArray();
    }

    public function getEngagementOfEachRole(string $contentType): array
    {
        $queryLikes = Reaction::query()
            ->whereType('like')
            ->whereReactionableType(type_to_class($contentType))
            ->whereBetween('created_at', [$this->firstDay, $this->lastDay])
            ->with('authenticationUsers.roles')
            ->get()
            ->flatMap(fn ($reaction) => $reaction->authenticationUsers
                ? $reaction->authenticationUsers->roles->pluck('name')
                : collect())
            ->countBy();

        $roleCommentsCount = Comments::query()
            ->join('inside_users', 'inside_content_comments.author', '=', 'inside_users.uuid')
            ->join('inside_users_roles', 'inside_users.uuid', '=', 'inside_users_roles.user_uuid')
            ->join('inside_roles', 'inside_users_roles.role_id', '=', 'inside_roles.id')
            ->whereBetween('inside_content_comments.created_at', [$this->firstDay, $this->lastDay])
            ->whereHas($contentType)
            ->select('inside_roles.name as role_name')
            ->get()
            ->pluck('role_name')
            ->countBy();

        $users = User::with('roles')
            ->where('status', 1)
            ->get()
            ->flatMap(fn ($user) => $user->roles->pluck('name'))
            ->countBy();

        $allRoles = $users->keys();

        $mergedStats = $allRoles->mapWithKeys(function ($role) use ($users, $queryLikes, $roleCommentsCount) {
            return [
                $role => [
                    'users' => $users->get($role, 0),
                    'likes' => $queryLikes->get($role, 0),
                    'comments' => $roleCommentsCount->get($role, 0),
                ],
            ];
        });

        return $mergedStats->sortByDesc(fn ($stats) => $stats['likes'] + $stats['comments'])->toArray();
    }

    public function getTop10MostViewedContents(string $langcode, array $contentTypes): Collection
    {
        $statisticsRepository = new StatisticsBetaRepository();
        $topViewedContent = $statisticsRepository->getMostViewedContents(
            [
                'types' => $contentTypes,
                'from' => $this->firstDay,
                'to'   => $this->lastDay,
                'langcode' => $langcode,
            ]
        );

        $topViewedContent->transform(function ($item) {
            $attributes = collect($item->getAttributes())->except(['content_uuid']);
            $mergedAttributes = collect($item->content->getAttributes())
                ->merge($attributes)
                ->toArray();
            $mergedAttributes['slug'] = $item->content->getSlugAttribute();

            return $mergedAttributes;
        });

        return $topViewedContent;
    }

    public function getTop5MostLikedContents(string $langcode, array $contentTypes): Collection
    {
        $statisticsRepository = new StatisticsBetaRepository();
        $topLikedContent = $statisticsRepository->getMostLikedContents(
            [
                'types' => $contentTypes,
                'from' => $this->firstDay,
                'to'   => $this->lastDay,
                'langcode' => $langcode,
            ]
        );
        $topLikedContent->transform(function ($item) {
            $attributes = collect($item->getAttributes())->except(['content_uuid']);
            $mergedAttributes = collect($item->content->getAttributes())
                ->merge($attributes)
                ->toArray();
            $mergedAttributes['slug'] = $item->content->getSlugAttribute();

            return $mergedAttributes;
        });

        return $topLikedContent;
    }

    public function getTop5MostCommentedContents(string $langcode, array $contentTypes): Collection
    {
        $statisticsRepository = new StatisticsBetaRepository();
        $topCommentedContents = $statisticsRepository->getMostCommentedContents(
            [
                'types' => $contentTypes,
                'from' => $this->firstDay,
                'to'   => $this->lastDay,
                'langcode' => $langcode,
            ]
        );

        $topCommentedContents->transform(function ($item) {
            $attributes = collect($item->getAttributes())->except(['content_uuid']);
            $mergedAttributes = collect($item->content->getAttributes())
                ->merge($attributes)
                ->toArray();
            $mergedAttributes['slug'] = $item->content->getSlugAttribute();

            return $mergedAttributes;
        });

        return $topCommentedContents;
    }

    public function getContentsViewedOnlyByAuthor(string $langcode, array $contentTypes): Collection
    {
        $statisticsRepository = new StatisticsBetaRepository();
        $authorOnlyViewedContents = $statisticsRepository->getAuthorOnlyViewedContents(
            [
                'types' => $contentTypes,
                'from' => $this->firstDay,
                'to'   => $this->lastDay,
                'langcode' => $langcode,
            ]
        );
        $authorOnlyViewedContents->transform(function ($item) {
            $attributes = collect($item->getAttributes())->except(['content_uuid']);
            $mergedAttributes = collect($item->content->getAttributes())
                ->merge($attributes)
                ->toArray();
            $mergedAttributes['slug'] = $item->content->getSlugAttribute();

            return $mergedAttributes;
        });

        return $authorOnlyViewedContents;
    }

    public function getTop10MostViewedCategories(string $langcode, string $contentType, string $contentTypeCategory): Collection
    {
        $statisticsRepository = new StatisticsBetaRepository();
        $baseQuery = $statisticsRepository->getContentStatsQuery(
            $contentType,
            [
                'from' => $this->firstDay,
                'to' => $this->lastDay,
                'langcode' => $langcode,
            ]
        );
        $categoryTable = type_to_table($contentTypeCategory);
        $categoryClass = type_to_class($contentTypeCategory);
        $contentClass = type_to_class($contentType);

        return DB::query()
            ->fromSub($baseQuery, 'content_stats')
            ->join('inside_pivots', function ($join) use ($categoryClass, $contentClass) {
                $join->on('inside_pivots.parent_uuid', '=', 'content_stats.content_uuid')
                    ->where('inside_pivots.parent_type', $contentClass)
                    ->where('inside_pivots.related_type', $categoryClass);
            })
            ->join($categoryTable, $categoryTable.'.uuid', '=', 'inside_pivots.related_uuid')
            ->selectRaw("
                {$categoryTable}.uuid   AS category_uuid,
                {$categoryTable}.title  AS category_title,
                SUM(content_stats.total_views)   AS total_views,
                SUM(content_stats.unique_views)  AS unique_views
            ")
            ->groupBy(
                "{$categoryTable}.uuid",
                "{$categoryTable}.title"
            )
            ->orderByDesc('total_views')
            ->get();
    }

    public function getActivityStatistics(string $langcode): array
    {
        return [
            [
                'type'  => 'likes',
                'value' => Reaction::query()
                    ->whereType('like')
                    ->where('langcode', $langcode)
                    ->whereBetween('created_at', [$this->firstDay, $this->lastDay])
                    ->count(),
            ],
            [
                'type'  => 'notifications_email',
                'value' => NotificationLog::query()
                    ->where('channel', 'email')
                    ->whereBetween('created_at', [$this->firstDay, $this->lastDay])
                    ->count(),
            ],
            [
                'type'  => 'comments',
                'value' => Comments::query()
                    ->where('status', 1)
                    ->where('langcode', $langcode)
                    ->whereBetween('inside_content_comments.created_at', [$this->firstDay, $this->lastDay])
                    ->count(),
            ],
            [
                'type'  => 'notifications_web',
                'value' => NotificationLog::query()
                    ->where('channel', 'web')
                    ->whereBetween('created_at', [$this->firstDay, $this->lastDay])
                    ->count(),
            ],
        ];
    }

    public function getTopViewedContent(int $monthId, string $langcode, string $classType = null, int $limit = 10)
    {
        return $this->getTopContentByMetric(
            $monthId,
            'inside_advanced_stats_top_viewed_contents',
            'total_views',
            $langcode,
            $classType,
            $limit
        );
    }

    public function getTopLikedContent(int $monthId, string $langcode, string $classType = null, int $limit = 10)
    {
        return $this->getTopContentByMetric(
            $monthId,
            'inside_advanced_stats_top_liked_contents',
            'likes',
            $langcode,
            $classType,
            $limit
        );
    }

    public function getRoleCommitmentStatistics(int $monthId, string $classType = null)
    {
        $query = DB::table('inside_advanced_stats_engagement')
            ->select(
                'role_name',
                DB::raw('SUM(users_nbr) as total_users'),
                DB::raw('SUM(likes) as total_likes'),
                DB::raw('SUM(comments) as total_comments')
            )
            ->where('month_id', $monthId)
            ->groupBy('role_name');

        if ($classType) {
            $query->where('class_type', $classType);
        }

        $rows = $query
            ->orderByDesc(DB::raw('SUM(likes + comments)'))
            ->get();

        return $rows->map(fn ($row) => [
            'label' => ucfirst(Role::getHumanName($row->role_name))." ({$row->total_users})",
            'likes' => (int) $row->total_likes,
            'comments' => (int) $row->total_comments,
        ])->values();
    }

    public function getBottomContent(int $monthId, string $langcode, string $classType = null, int $limit = 3)
    {
        $query = DB::table('inside_advanced_viewed_only_by_author')
            ->where('langcode', $langcode)
            ->where('month_id', $monthId);

        if ($classType) {
            $query->where('class_type', $classType);
        }

        $rows = $query->limit($limit)->get();

        return $rows->map(function ($row) {
            $slugs = [];

            try {
                if (class_exists($row->class_type)) {
                    $model = $row->class_type::query()->find($row->uuid);
                    if ($model && method_exists($model, 'getSlugAttribute')) {
                        $slugs = $model->getSlugAttribute();
                    }
                }
            } catch (\Throwable $e) {
            }

            return [
                'title' => $row->title,
                'uuid' => $row->uuid,
                'content_type' => class_to_type($row->class_type),
                'slug' => $slugs,
            ];
        })->values();
    }

    public function getTopCommentedContent(int $monthId, string $langcode, string $classType = null, int $limit = 10)
    {
        return $this->getTopContentByMetric(
            $monthId,
            'inside_advanced_stats_top_commented_contents',
            'comments',
            $langcode,
            $classType,
            $limit
        );
    }

    private function getTopContentByMetric(
        int $monthId,
        string $tableName,
        string $metricColumn,
        string $langcode,
        string $classType = null,
        int $limit = 50
    ) {
        $query = DB::table($tableName)
            ->where('langcode', $langcode)
            ->where('month_id', $monthId);

        if ($classType) {
            $query->where('class_type', $classType);
        }

        $rows = $query->orderBy($metricColumn, 'desc')
            ->limit($limit)
            ->get();

        return $rows->map(function ($row) use ($metricColumn) {
            $slugs = [];

            try {
                if (class_exists($row->class_type)) {
                    $model = $row->class_type::query()->find($row->uuid);
                    if ($model && method_exists($model, 'getSlugAttribute')) {
                        $slugs = $model->getSlugAttribute();
                    }
                }
            } catch (\Throwable $e) {
            }

            return [
                'title' => $row->title,
                'uuid' => $row->uuid,
                'content_type' => class_to_type($row->class_type),
                'slug' => $slugs,
                'value' => (int) $row->$metricColumn,
            ];
        })->values();
    }

    public function getTopCategories(int $monthId, string $langcode, string $classType = null, int $limit = 5)
    {
        $query = DB::table('inside_advanced_stats_top_viewed_categories')
            ->where('langcode', $langcode)
            ->where('month_id', $monthId);

        if ($classType) {
            $query->where('class_type', $classType);
        }

        $rows = $query->orderBy('total_views', 'desc')
            ->limit($limit)
            ->get();

        return $rows->map(function ($row) {
            return [
                'title' => $row->title,
                'uuid' => $row->uuid,
                'content_type' => class_to_type($row->class_type),
                'value' => (int) $row->total_views,
            ];
        })->values();
    }

    public function overridePeriod(string $firstDay, string $lastDay): self
    {
        $from = Carbon::parse($firstDay)->startOfDay();
        $to = Carbon::parse($lastDay)->endOfDay();

        if ($from->greaterThan($to)) {
            throw new \InvalidArgumentException('First day must be before last day.');
        }

        $this->firstDay = $from->toDateString();
        $this->lastDay = $to->toDateString();

        return $this;
    }

    public function computeAdvancedStatistics(): void
    {
        $periodDate = Carbon::parse($this->firstDay);

        $month = $periodDate->month;
        $year = $periodDate->year;

        $monthExists = DB::table('inside_advanced_stats_months')
            ->where('month', $month)
            ->where('year', $year)
            ->exists();

        if ($monthExists) {
            throw new \Exception('Month/year already exists');
        }

        $monthId = DB::table('inside_advanced_stats_months')->insertGetId([
            'month' => $month,
            'year' => $year,
        ]);

        // Attendance
        $attendanceStats = $this->getAttendenceStats();
        DB::table('inside_advanced_stats_attendance')->insert([
            'month_id' => $monthId,
            ...$attendanceStats,
        ]);

        // Activity + Contents
        collect(list_languages())->each(function ($langcode) use ($monthId) {
            // Activities
            $activities = $this->getActivityStatistics($langcode);
            // Administrations
            $administrations = collect(config('statistics.types'))
                ->keys()
                ->map(function ($contentType) use ($langcode) {
                    return [
                        'content_type' => $contentType,
                        'value' => type_to_class($contentType)::query()
                            ->where('langcode', $langcode)
                            ->whereBetween('created_at', [$this->firstDay, $this->lastDay])
                            ->count(),
                    ];
                })
                ->values();

            DB::table('inside_advanced_stats_general')->insert([
                'month_id' => $monthId,
                'langcode' => $langcode,
                'data'     => json_encode([
                    'activity'      => $activities,
                    'administration' => $administrations,
                ]),
            ]);
        });

        // Activity-Role-Based
        $activities = $this->getActivityRoleBasedStats();
        $activitiesFormated = collect($activities)->map(function ($data, $role) use ($monthId) {
            return [
                'month_id'          => $monthId,
                'role_name'         => $role,
                'users_nbr'         => $data['count'],
                'access_percentage' => $data['percentage'],
            ];
        })->values()->toArray();
        DB::table('inside_advanced_stats_activity_by_role')->insert($activitiesFormated);

        // Engagement
        collect(config('statistics.types'))
            ->filter(fn ($item) => $item['comments'] || $item['likes']) // only keep types with activity
            ->keys()
            ->mapWithKeys(function ($contentType) use ($monthId) {
                $rolesEngagement = $this->getEngagementOfEachRole($contentType);

                $rows = collect($rolesEngagement)->map(fn ($roleData, $roleName) => [
                    'month_id'   => $monthId,
                    'class_type' => type_to_class($contentType),
                    'role_name'  => $roleName,
                    'users_nbr'  => $roleData['users'] ?? 0,
                    'likes'      => $roleData['likes'] ?? 0,
                    'comments'   => $roleData['comments'] ?? 0,
                ])->toArray();

                DB::table('inside_advanced_stats_engagement')->insert($rows);
            });

        // TOP 10 des contenus les plus vus
        collect(config('statistics.types'))
            ->keys()
            ->mapWithKeys(function ($contentType) use ($monthId) {
                collect(list_languages())->each(function ($langcode) use ($contentType, $monthId) {
                    $topViewedContents = $this->getTop10MostViewedContents($langcode, [$contentType]);

                    $rows = $topViewedContents
                        ->values()
                        ->map(function ($item) use ($monthId) {
                            return [
                                'month_id'    => $monthId,
                                'title'       => $item['title'],
                                'class_type'  => type_to_class($item['type']),
                                'uuid'        => $item['uuid'],
                                'langcode'    => $item['langcode'],
                                'total_views' => $item['total_views'],
                            ];
                        })
                        ->toArray();

                    DB::table('inside_advanced_stats_top_viewed_contents')->insert($rows);
                });
            });

        // TOP 5 des contenus les plus liké
        collect(config('statistics.types'))
            ->filter(fn ($item) => $item['likes'])
            ->keys()
            ->mapWithKeys(function ($contentType) use ($monthId) {
                collect(list_languages())->each(function ($langcode) use ($contentType, $monthId) {
                    $topLikedContent = $this->getTop5MostLikedContents($langcode, [$contentType]);

                    $rows = $topLikedContent
                        ->values()
                        ->map(function ($item) use ($monthId) {
                            return [
                                'month_id'    => $monthId,
                                'title'       => $item['title'],
                                'class_type'  => type_to_class($item['type']),
                                'uuid'        => $item['uuid'],
                                'langcode'    => $item['langcode'],
                                'likes'       => $item['likes'],
                            ];
                        })
                        ->toArray();

                    DB::table('inside_advanced_stats_top_liked_contents')->insert($rows);
                });
            });

        collect(config('statistics.types'))
            ->filter(fn ($item) => $item['comments'])
            ->keys()
            ->mapWithKeys(function ($contentType) use ($monthId) {
                collect(list_languages())->each(function ($langcode) use ($contentType, $monthId) {
                    $topCommentedContent = $this->getTop5MostCommentedContents($langcode, [$contentType]);

                    $rows = $topCommentedContent
                        ->values()
                        ->map(function ($item) use ($monthId) {
                            return [
                                'month_id'    => $monthId,
                                'title'       => $item['title'],
                                'class_type'  => type_to_class($item['type']),
                                'uuid'        => $item['uuid'],
                                'langcode'    => $item['langcode'],
                                'comments'    => $item['comments_count'],
                            ];
                        })
                        ->toArray();

                    DB::table('inside_advanced_stats_top_commented_contents')->insert($rows);
                });
            });

        // contenus consultés uniquement par l'auteur
        collect(config('statistics.types'))
            ->keys()
            ->mapWithKeys(function ($contentType) use ($monthId) {
                collect(list_languages())->each(function ($langcode) use ($contentType, $monthId) {
                    $contentsViewedOnlyByAuthor = $this->getContentsViewedOnlyByAuthor($langcode, [$contentType]);

                    $rows = $contentsViewedOnlyByAuthor
                        ->values()
                        ->map(function ($item) use ($monthId) {
                            return [
                                'month_id'    => $monthId,
                                'title'       => $item['title'],
                                'class_type'  => type_to_class($item['type']),
                                'uuid'        => $item['uuid'],
                                'langcode'    => $item['langcode'],
                            ];
                        })
                        ->toArray();

                    DB::table('inside_advanced_viewed_only_by_author')->insert($rows);
                });
            });

        collect(config('statistics.types'))
            ->filter(fn ($item) => $item['category'] !== null)
            ->mapWithKeys(function ($config, $contentType) use ($monthId) {
                collect(list_languages())->each(function ($langcode) use ($contentType, $config, $monthId) {
                    $mostViewedCategories = $this->getTop10MostViewedCategories($langcode, $contentType, $config['category']);

                    $rows = $mostViewedCategories
                        ->values()
                        ->map(function ($item) use ($monthId, $contentType, $langcode) {
                            return [
                                'month_id'    => $monthId,
                                'title'       => $item->category_title,
                                'class_type'  => type_to_class($contentType),
                                'uuid'        => $item->category_uuid,
                                'langcode'    => $langcode,
                                'total_views' => $item->total_views,
                                'unique_views' => $item->unique_views,
                            ];
                        })
                        ->toArray();

                    DB::table('inside_advanced_stats_top_viewed_categories')->insert($rows);
                });
            });
    }

    private function formatAttendance($row)
    {
        return [
            [
                'type'  => 'users_has_role_authenticated',
                'value' => (int) $row->users_has_role_authenticated,
            ],
            [
                'type'  => 'logged_users_period',
                'value' => (int) $row->authenticated_nbr,
            ],
            [
                'type'   => 'authenticated_users_login_rate',
                'value'  => (float) $row->authenticated_percentage,
                'suffix' => '%',
            ],
            [
                'type'  => 'site_visitors_period',
                'value' => (int) $row->access_nbr,
            ],
            [
                'type'   => 'authenticated_users_visit_rate',
                'value'  => (float) $row->access_percentage,
                'suffix' => '%',
            ],
        ];
    }
}
