<?php

namespace Inside\Statistics\Repositories;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Authentication\Models\Token;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Section;
use Inside\Database\Eloquent\Builder;
use Inside\Permission\Models\Role;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Contracts\Statistic as StatisticContract;
use Inside\Statistics\Facades\Stats;
use Inside\Statistics\Models\AuthenticationStatistic;
use Inside\Statistics\Models\ContentStatistic;
use Inside\Statistics\Models\GlobalStatistic;
use Inside\Statistics\Models\Statistic;
use Inside\Statistics\Models\UserStatistic;
use Inside\Support\Str;
use Throwable;

class StatisticsRepository
{
    public function neverAuthenticatedOnlyFiltered(array $filters = []): bool
    {
        return ! isset($filters['never_authenticated'])
          || $filters['never_authenticated'] === true
          || in_array($filters['never_authenticated'], [1, '1', 'true']);
    }

    protected function getFromDateFromFilters(array $filters = []): ?Carbon
    {
        $fromDate = null;
        if (isset($filters['from'])) {
            $fromDate = $this->getCarbonFromFront($filters['from']);
        }

        return $fromDate;
    }

    public function getCarbonFromFront(string $date): ?Carbon
    {
        try {
            return get_date($date, 'Y-m-d')->setTime(12, 0);
        } catch (Throwable $exception) {
            Log::error('[StatisticsRepository::getCarbonFromFront] Failed to parse date from from => '.$exception->getMessage());
        }

        return null;
    }

    protected function getToDateFromFilters(array $filters = []): ?Carbon
    {
        $toDate = null;
        if (isset($filters['from'])) {
            if (isset($filters['to'])) {
                $toDate = $this->getCarbonFromFront($filters['to']);
            }
            if ($toDate === null) {
                $toDate = now();
            }
        }

        return $toDate;
    }

    public function getContentStatsCount(string $type, array $filters = []): int
    {
        $categoryId = null;
        $langCode = null;
        $config = config('statistics.types.'.$type, []);
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);

        if (isset($config['category']) && isset($filters['category_uuid'])) {
            $categoryId = $filters['category_uuid'];
        }
        if (isset($filters['langcode'])) {
            $langCode = $filters['langcode'];
        }

        $countQuery = ContentStatistic::where('type', $type)->distinct();
        if ($fromDate !== null) {
            $countQuery->whereDate('content_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $countQuery->whereDate('content_statistics.date', '<=', $toDate->toDateString());
        }
        if ($categoryId !== null) {
            $countQuery->where('category_uuid', $categoryId);
        }
        if ($langCode !== null) {
            $countQuery->where('langcode', $langCode);
        }

        return $countQuery->count('uuid');
    }

    public function getContentStatsQuery(string $type, array $filters = []): Builder
    {
        $categoryIds = null;
        $langCode = null;
        $config = config('statistics.types.'.$type, []);
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);

        if (isset($config['category']) && isset($filters['category_uuid'])) {
            $categoryIds = $filters['category_uuid'];
            if (! is_array($categoryIds)) {
                $categoryIds = [$categoryIds];
            }
        }

        if (isset($filters['langcode'])) {
            $langCode = $filters['langcode'];
        }
        $casts = [
            'visits' => 'integer',
            'unique_visits' => 'integer',
            'status' => 'boolean',
            'has_statistics' => 'boolean',
            'published_at' => 'timestamp',
        ];

        $query = ContentStatistic::where('type', $type)
          ->distinct()
          ->select([
              'uuid',
          ]);
        if ($fromDate !== null) {
            $query->whereDate('content_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('content_statistics.date', '<=', $toDate->toDateString());
        }
        if ($categoryIds !== null) {
            $query->whereIn('category_uuid', $categoryIds);
        }
        if ($langCode !== null) {
            $query->where('langcode', $langCode);
        }
        $query->addSelect([
            'title' => ContentStatistic::from('content_statistics as content_statistics_title')
              ->select('content_statistics_title.title as title')
              ->where('content_statistics_title.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_title.uuid')
              ->orderByDesc('content_statistics_title.date')
              ->take(1),
            'slug' => ContentStatistic::from('content_statistics as content_statistics_slug')
              ->select('content_statistics_slug.slug as slug')
              ->where('content_statistics_slug.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_slug.uuid')
              ->orderByDesc('content_statistics_slug.date')
              ->take(1),
            'created_at' => ContentStatistic::from('content_statistics as content_statistics_created_at')
              ->select('content_statistics_created_at.created_at as created_at')
              ->where('content_statistics_created_at.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_created_at.uuid')
              ->orderByDesc('content_statistics_created_at.date')
              ->take(1),
            'updated_at' => ContentStatistic::from('content_statistics as content_statistics_updated_at')
              ->select('content_statistics_updated_at.updated_at as updated_at')
              ->where('content_statistics_updated_at.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_updated_at.uuid')
              ->orderByDesc('content_statistics_updated_at.date')
              ->take(1),
            'published_at' => ContentStatistic::from('content_statistics as content_statistics_published_at')
              ->select('content_statistics_published_at.published_at as published_at')
              ->where('content_statistics_published_at.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_published_at.uuid')
              ->orderByDesc('content_statistics_published_at.date')
              ->take(1),
            'status' => ContentStatistic::from('content_statistics as content_statistics_status')
              ->select('content_statistics_status.status as status')
              ->where('content_statistics_status.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_status.uuid')
              ->orderByDesc('content_statistics_status.date')
              ->take(1),
            'langcode' => ContentStatistic::from('content_statistics as content_statistics_langcode')
              ->select('content_statistics_langcode.langcode as langcode')
              ->where('content_statistics_langcode.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_langcode.uuid')
              ->orderByDesc('content_statistics_langcode.date')
              ->take(1),
            'category_title' => ContentStatistic::from('content_statistics as content_statistics_category_title')
              ->select('content_statistics_category_title.category_title as category_title')
              ->where('content_statistics_category_title.type', $type)
              ->whereColumn('content_statistics.uuid', 'content_statistics_category_title.uuid')
              ->orderByDesc('content_statistics_category_title.date')
              ->take(1),
            'visits' => DB::raw('SUM(visits) AS visits'),
            'unique_visits' => DB::raw('SUM(unique_visits) AS unique_visits'),
            'content_type' => DB::raw('type AS content_type'),
            'has_statistics' => DB::raw('COUNT(id) AS has_statistics'),
        ]);
        if (isset($config['likes']) && $config['likes']) {
            $query->addSelect([
                'likes' => DB::raw('SUM(likes) AS likes'),
            ]);
            $casts['likes'] = 'integer';
        }
        if (isset($config['comments']) && $config['comments']) {
            $query->addSelect([
                'comments' => DB::raw('SUM(comments) AS comments'),
            ]);
            $casts['comments'] = 'integer';
        }

        return $query->withCasts($casts)->groupBy([
            'content_statistics.uuid',
            'content_statistics.type',
        ]);
    }

    public function getUserVisitStatsCount(string $type, array $filters = [], ?string $search = null): int
    {
        $query = $this->getUserVisitStatsQuery($type, $filters, $search);

        return DB::table(DB::raw("({$query->toSql()}) as sub"))
          ->mergeBindings($query->getQuery())
          ->count();
    }

    public function getUserVisitStatsQuery(string $type, array $filters = [], ?string $search = null): Builder
    {
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);
        $casts = [
            'visits' => 'integer',
            'unique_visits' => 'integer',
        ];

        $query = UserStatistic::where('type', $type)
          ->distinct()
          ->select(
              [
                  'uuid',
              ]
          );
        if ($fromDate !== null) {
            $query->whereDate('user_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('user_statistics.date', '<=', $toDate->toDateString());
        }
        $statisticService = App::make(StatisticContract::class);
        $query->addSelect([
            'visits' => DB::raw('SUM(user_statistics.visits) AS visits'),
            'unique_visits' => DB::raw('SUM(user_statistics.unique_visits) AS unique_visits'),
            'email' => User::select('email')
              ->whereColumn('user_statistics.uuid', 'inside_users.uuid')
              ->latest()
              ->take(1),
            'full_name' => Users::select(DB::raw($statisticService->getFullNameSelectRawQuery()))
              ->whereColumn('user_statistics.uuid', 'inside_content_users.uuid')
              ->latest()
              ->take(1),
        ])
          ->whereHas('users', fn ($query) => $query->where('is_maintenance', false))
          ->when(! is_null($search) && ! empty(trim($search)), function ($query) use ($search) {
              $query->whereHas(
                  'users',
                  fn ($query) => $query
                    ->whereLike(DB::raw('CONCAT(firstname, CONCAT(\' \', lastname))'), '%'.trim($search ?? '').'%')
                    ->orWhereLike('email', '%'.trim($search ?? '').'%')
              );
          });

        return $query->withCasts($casts)->groupBy('uuid');
    }

    public function getNeverAuthenticatedOnlyStats(Builder $subQuery, array $filters): Builder
    {
        $querySearch = $filters['query'] ?? null;
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);

        $query = User::whereNotIn('inside_users.uuid', $subQuery)
          ->select('inside_users.*');

        $query->where(function ($query) use ($fromDate, $toDate) {
            if ($fromDate !== null) {
                $query->whereDate('inside_users.last_login_at', '<', $fromDate);
            }
            if ($toDate !== null) {
                $query->orWhereDate('inside_users.last_login_at', '>', $toDate);
            }
        });

        $query->join(
            'inside_content_users as cu',
            'cu.uuid',
            'inside_users.uuid'
        )->whereHas('information', function ($query) use ($querySearch) {
            $query->where('is_maintenance', false);
            if ($querySearch) {
                $query->whereRawLike("CONCAT(firstname, ' ', lastname)", "%$querySearch%")
                  ->orWhereLike('email', "%$querySearch%");
            }
        })->addSelect([
            'full_name' => DB::raw("CONCAT(UPPER(LEFT(firstname, 1)), SUBSTRING(LOWER(firstname), 2), ' ', UPPER(cu.lastname)) AS full_name"),
            'last_login_at_timestamp' => Token::select('created_at')
              ->whereColumn(
                  'user_uuid',
                  'inside_users.uuid'
              )
              ->latest()
              ->take(1),
        ]);

        return $query;
    }

    public function getAuthenticationStatsQuery(array $filters = []): Builder
    {
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);

        $query = AuthenticationStatistic::select([
            DB::raw('SUM(authentication_statistics.connection_count) AS connection_count'),
            'authentication_statistics.uuid',
            'authentication_statistics.full_name',
            'authentication_statistics.email',
            'authentication_statistics.roles',
            'authentication_statistics.enabled',
            'authentication_statistics.last_login_at_timestamp',
        ]);

        if ($fromDate !== null) {
            $query->whereDate(
                'authentication_statistics.date',
                '>=',
                $fromDate->format('Y-m-d')
            );
        }
        if ($toDate !== null) {
            $query->whereDate(
                'authentication_statistics.date',
                '<=',
                $toDate->format('Y-m-d')
            );
        }

        return $query->withCasts([
            'last_login_at_timestamp' => 'timestamp',
            'enabled' => 'boolean',
            'email' => 'string',
        ])->groupBy([
            'authentication_statistics.uuid',
            'authentication_statistics.full_name',
            'authentication_statistics.email',
            'authentication_statistics.roles',
            'authentication_statistics.enabled',
        ]);
    }

    public function getContentStatQuery(
    string $contentType,
    string $type = 'view',
    ?Carbon $fromDate = null,
    ?Carbon $toDate = null,
    ?string $langCode = null
  ): Builder {
        $query = Statistic::where('statisticable_type', type_to_class($contentType))
          ->where('inside_statistics.type', $type)
          ->join(
              type_to_table($contentType),
              type_to_table($contentType).'.uuid',
              '=',
              'statisticable_uuid'
          )
          ->where(type_to_table($contentType).'.status', '=', true);
        if ($langCode !== null) {
            $query->where('inside_statistics.langcode', $langCode);
        }
        if ($fromDate !== null) {
            $query->where(type_to_table($contentType).'.created_at', '>=', $fromDate);
        }
        if ($toDate !== null) {
            $query->where(type_to_table($contentType).'.created_at', '<=', $toDate);
        }

        return $query;
    }

    public function getReactionCountQuery(string $type, ?string $langCode = null): Builder
    {
        return Reaction::whereType($type)
          ->where(function ($query) use ($langCode) {
              foreach (Schema::getContentTypes() as $type) {
                  $query->orWhere(function ($query) use ($type, $langCode) {
                      if (Str::endsWith($type, 'menus')) {
                          return;
                      }
                      $query->where('inside_reactions.reactionable_type', type_to_class($type))
                        ->whereExists(function ($query) use ($type, $langCode) {
                            $query->select(type_to_table($type).'.*')
                              ->from(type_to_table($type))
                              ->whereStatus(true)
                              ->whereColumn('uuid', 'inside_reactions.reactionable_uuid');
                            if ($langCode !== null) {
                                $query->where(type_to_table($type).'.langcode', $langCode);
                            }
                        });
                  });
              }
          });
    }

    public function getReactionStatQuery(
    string $type,
    ?Carbon $fromDate = null,
    ?Carbon $toDate = null,
    ?string $langCode = null
  ): Builder {
        $query = Statistic::where('statisticable_type', Reaction::class)
          ->where('inside_statistics.type', 'reaction.'.$type)
          ->join(
              'inside_reactions',
              'inside_reactions.id',
              '=',
              'statisticable_uuid'
          )
          ->when($fromDate !== null, function ($query) use ($fromDate) {
              $query->where('inside_reactions.created_at', '>=', $fromDate);
          })
          ->when($toDate !== null, function ($query) use ($toDate) {
              $query->where('inside_reactions.created_at', '<=', $toDate);
          })
          ->where(function ($query) use ($langCode) {
              foreach (Schema::getContentTypes() as $type) {
                  $query->orWhere(function ($query) use ($type, $langCode) {
                      if (Str::endsWith($type, 'menus')) {
                          return;
                      }
                      $query->where('inside_reactions.reactionable_type', type_to_class($type))
                        ->whereExists(function ($query) use ($type, $langCode) {
                            $query->select(type_to_table($type).'.*')->from(type_to_table($type))
                              ->whereStatus(true)
                              ->whereColumn('uuid', 'inside_reactions.reactionable_uuid');
                            if ($langCode !== null) {
                                $query->where(type_to_table($type).'.langcode', $langCode);
                            }
                        });
                  });
              }
          });
        if ($langCode !== null) {
            $query->where('inside_statistics.langcode', $langCode);
        }

        return $query;
    }

    public function getContentReadersQuery(Content | Section $content, array $filters, ?string $search = null): Builder
    {
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);
        $query = Statistic::select([
            'user_uuid',
            'full_name' => Users::selectRaw(Stats::getFullNameSelectRawQuery())
              ->whereColumn('uuid', 'inside_statistics.user_uuid')
              ->take(1),
            'email' => Users::select('email')
              ->whereColumn('uuid', 'inside_statistics.user_uuid')
              ->take(1),
            'image' => Users::select('image')
              ->whereColumn('uuid', 'inside_statistics.user_uuid')
              ->take(1),
        ])
          ->where('statisticable_type', get_class($content))
          ->where('statisticable_uuid', $content->uuid)
          ->whereType('view')
          ->whereHas('user.information', fn (Builder $query) => $query->where('is_maintenance', false))
          ->when($fromDate, fn (Builder $query) => $query->whereDate('created_at', '>=', $fromDate->toDateString()))
          ->when($toDate, fn (Builder $query) => $query->whereDate('created_at', '<=', $toDate->toDateString()))
          ->when(! is_null($search), fn (Builder $query) => $query->whereHas(
              relation: 'user.information',
              callback: fn (Builder $query) => $query->whereRawLike(Stats::getFullNameSelectRawQuery(), Str::wrap($search, '%'))
          ))
          ->groupBy('inside_statistics.user_uuid')
          ->orderBy('full_name');

        return $query;
    }

    public function getUserOnlyViewedContentListQuery(User $user, array $types, array $filters = [], ?string $search = null): Builder
    {
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);

        return Statistic::select([
            'user_uuid',
            'type',
            'statisticable_uuid',
            'statisticable_type',
            DB::raw('count(user_uuid) as viewed'),
        ])
          ->with('statisticable')
          ->where('user_uuid', $user->uuid)
          ->whereIn('statisticable_type', array_map('type_to_class', $types))
          ->whereHasMorph('statisticable', array_map('type_to_class', $types))
          ->whereType('view')
          ->when(! is_null($fromDate), fn (Builder $query) => $query->whereDate('created_at', '>=', $fromDate->toDateString()))
          ->when(! is_null($toDate), fn (Builder $query) => $query->whereDate('created_at', '<=', $toDate->toDateString()))
          ->when(
              ! is_null($search),
              fn (Builder $query) => $query->whereHasMorph(
                  relation: 'statisticable',
                  types: array_map('type_to_class', $types),
                  callback: fn (Builder $query) => $query->whereLike('title', '%'.$search.'%')
              )
          )
          ->groupBy([
              'user_uuid',
              'type',
              'statisticable_uuid',
              'statisticable_type',
          ])
          ->orderBy('viewed', 'desc');
    }

    public function getUserOnlyViewedContentList(User $user, array $types, array $filters = [], ?string $search = null): Collection
    {
        return $this->getUserOnlyViewedContentListQuery($user, $types, $filters, $search)->get()->map(fn (Statistic $statistic) => [
            'type' => $statistic->type,
            'uuid' => $statistic->statisticable_uuid,
            'title' => $statistic->statisticable->title,
            'content_type' => $statistic->statisticable_type,
            'viewed' => $statistic->viewed,
            'is_viewed' => 1,
        ]);
    }

    public function getUserViewedContentListQuery(User $user, array $types, array $filters, Collection $viewed, ?string $search): Collection
    {
        $result = $viewed;
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);

        foreach ($types as $type) {
            if (! Schema::hasContentType($type)) {
                continue;
            }

            $query = call_user_func(type_to_class($type).'::query')
              ->whereStatus(true)
              ->when($fromDate !== null, fn (Builder $query) => $query->whereDate('created_at', '>=', $fromDate->toDateString()))
              ->when($toDate !== null, fn (Builder $query) => $query->whereDate('created_at', '<=', $toDate->toDateString()))
              ->when($search !== null, fn (Builder $query) => $query->whereLike('title', '%'.$search.'%'))
              ->whereNotIn('uuid', $viewed->where('type', $type)->pluck('uuid')->toArray());

            $result = $result->concat($query->get()->map(fn (Content $content) => [
                'uuid' => $content->uuid,
                'title' => $content->title,
                'content_type' => $type,
                'viewed' => 0,
                'is_viewed' => 0,
            ]));
        }

        return $result;
    }

    public function getGlobalStatQuery(array $filters = []): ?object
    {
        $fromDate = $this->getFromDateFromFilters($filters);
        $toDate = $this->getToDateFromFilters($filters);
        $query = GlobalStatistic::query();

        if ($fromDate !== null) {
            $query->whereDate('global_statistics.date', '>=', $fromDate->toDateString());
        }
        if ($toDate !== null) {
            $query->whereDate('global_statistics.date', '<=', $toDate->toDateString());
        }

        return $query->select([DB::raw('SUM(likes) AS likes'), DB::raw('SUM(comments) AS comments')])->first();
    }

    public function notReadFiltered(array $filters = []): bool
    {
        return ! isset($filters['not_read'])
          || $filters['not_read'] === true
          || in_array($filters['not_read'], [1, '1', 'true']);
    }
}
