<?php

namespace Inside\Statistics\Jobs;

use Carbon\Carbon;
use Illuminate\Support\Facades\Log;
use Inside\Jobs\Job;
use Inside\Statistics\Models\Statistic;
use Inside\Statistics\Models\UserStatistic;
use Inside\Statistics\Models\UserVisit;
use Symfony\Component\Console\Output\OutputInterface;

class ComputeUserVisitStatistics extends Job
{
    protected array $config = [];

    public int $timeout = 21600;

    public function __construct(
    protected ?Carbon $from = null,
    protected ?Carbon $to = null,
    protected bool $reset = false,
    protected ?OutputInterface $output = null
  ) {
    }

    public function handle(): void
    {
        $this->config = config('statistics');

        if ($this->reset) {
            UserStatistic::truncate();
            UserVisit::truncate();
        }

        if (is_null($this->to)) {
            $this->to = now()->setTime(23, 59, 59);
        }

        if ($this->from && ! $this->from->isBefore($this->to)) {
            return;
        }

        Log::info('Début de la compilation des statistiques de visite des utilisateurs');
        Log::info(str_repeat('#', 80));

        if ($this->reset) {
            Log::info('Mode réinitialisation activé');
        }

        if (! is_null($this->from)) {
            Log::info('De :'.$this->from->toDateString());
        }
        if (! is_null($this->to)) {
            Log::info('A :'.$this->to->toDateString());
        }

        $query = Statistic::where('type', 'view')
      ->when(! is_null($this->from), function ($query) {
          $query->whereDate('created_at', '>=', $this->from->toDateString());
      })
      ->when(! is_null($this->to), function ($query) {
          $query->whereDate('created_at', '<=', $this->to->toDateString());
      }, function ($query) {
          $query->whereDate('created_at', '<', now()->toDateString());
      })
      ->orderBy('created_at');

        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar($query->count());
            $bar->start();
        }
        Log::info($query->count().' entrée de statistiques à traiter');
        $query->each(
            function (Statistic $statistic) use ($bar) {
                if ($bar) {
                    $bar->advance();
                }
                if ($statistic->statisticable === null) {
                    return;
                }
                if ($statistic->user === null || $statistic->user->information === null
                || $statistic->user->information->is_maintenance
                ) {
                    return;
                }
                $contentType = class_to_type($statistic->statisticable_type);
                if (! array_key_exists(class_to_type($contentType), $this->config['types'])) {
                    return;
                }

                $this->computeUserStatisticForDate(
                    $statistic,
                    $contentType,
                    get_date((int) $statistic->created_at)->setTime(12, 0)
                );
            }
        );

        if ($bar) {
            $bar->finish();

            if ($this->output instanceof OutputInterface) {
                $this->output->writeln('');
                $this->output->writeln('<fg=cyan>'.str_repeat('#', 80).'</>');
            }
        }

        Log::info(str_repeat('#', 80));
    }

    protected function computeUserStatisticForDate(
    Statistic $statistic,
    string $contentType,
    Carbon $date
  ): void {
        $userStatistic = UserStatistic::firstOrCreate(
            [
                'uuid' => $statistic->user_uuid,
                'type' => $contentType,
                'date' => $date->toDateString(),
            ],
            [
                'visits' => 0,
                'unique_visits' => 0,
            ]
        );

        $userStatistic->increment('visits');

        $userVisit = UserVisit::firstOrCreate([
            'user_uuid' => $statistic->user_uuid,
            'uuid' => $statistic->statisticable_uuid,
            'type' => $contentType,
        ]);

        if ($userVisit->wasRecentlyCreated) {
            $userStatistic->increment('unique_visits');
        }
    }
}
