<?php

namespace Inside\Statistics\Jobs;

use Carbon\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Models\Contents\Comments;
use Inside\Jobs\Job;
use Inside\Reaction\Models\Reaction;
use Inside\Statistics\Models\ContentStatistic;
use Inside\Statistics\Models\ContentVisit;
use Inside\Statistics\Models\Statistic;
use Inside\Support\Str;
use Symfony\Component\Console\Output\OutputInterface;

class ComputeContentStatistics extends Job
{
    protected array $config;

    protected Collection $comments;

    public int $timeout = 21600;

    public function __construct(
    protected ?Carbon $from = null,
    protected ?Carbon $to = null,
    protected bool $reset = false,
    protected ?OutputInterface $output = null
  ) {
    }

    public function handle(): void
    {
        $this->config = config('statistics.types');
        $this->comments = collect();

        if ($this->reset) {
            ContentStatistic::truncate();
            ContentVisit::truncate();
        }

        if (is_null($this->to)) {
            $this->to = now()->setTime(23, 59, 59);
        }

        if ($this->from && ! $this->to->isAfter($this->from)) {
            return;
        }

        Log::info('Début de la compilation des statistiques de visite des contenus');
        Log::info(str_repeat('#', 80));

        if ($this->reset) {
            Log::info('Mode réinitialisation activé');
        }

        if (! is_null($this->from)) {
            Log::info('De :'.$this->from->toDateString());
        }
        if (! is_null($this->to)) {
            Log::info('A :'.$this->to->toDateString());
        }

        $query = Statistic::where('type', 'view')
            ->when(! is_null($this->from), function ($query) {
                $query->whereDate('created_at', '>=', $this->from->toDateString());
            })
            ->when(! is_null($this->to), function ($query) {
                $query->whereDate('created_at', '<=', $this->to->toDateString());
            }, function ($query) {
                $query->whereDate('created_at', '<', now()->toDateString());
            })
            ->orderBy('created_at');

        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar($query->count());
            $bar->start();
        }
        Log::info($query->count().' entrée de statistiques à traiter');
        $query->each(
            function (Statistic $statistic) use ($bar) {
                if ($bar) {
                    $bar->advance();
                }
                if ($statistic->statisticable === null) {
                    return;
                }
                if (
                    $statistic->user === null
                    || $statistic->user->information === null
                    || $statistic->user->information->is_maintenance
                ) {
                    return;
                }
                $contentType = class_to_type($statistic->statisticable_type);
                if (! array_key_exists(class_to_type($contentType), $this->config)) {
                    return;
                }
                $config = $this->config[$contentType];
                $this->computeContentStatisticForDate(
                    $statistic,
                    $contentType,
                    $config['category'],
                    get_date((int) $statistic->created_at)->setTime(12, 0),
                    isset($config['likes']) && $config['likes'],
                    isset($config['comments']) && $config['comments']
                );
            }
        );

        if ($bar) {
            $bar->finish();
            if ($this->output instanceof OutputInterface) {
                $this->output->writeln('');
                $this->output->writeln('<fg=cyan>'.str_repeat('#', 80).'</>');
            }
        }

        Log::info(str_repeat('#', 80));
    }

    /**
     * @param Statistic $statistic
     * @param string $contentType
     * @param string|null $categoryFieldName
     * @param Carbon $date
     * @param bool $withLikes
     * @param bool $withComments
     * @return void
     */
    protected function computeContentStatisticForDate(
    Statistic $statistic,
    string $contentType,
    ?string $categoryFieldName,
    Carbon $date,
    bool $withLikes = false,
    bool $withComments = false
  ): void {
        $category = null;
        if ($categoryFieldName !== null) {
            $category = $statistic->statisticable->{Str::camel($categoryFieldName)};
            if ($category instanceof Collection) {
                $category = $category->first();
            }
        }

        $userHasVisited = ContentVisit::where('type', $contentType)
      ->where('uuid', $statistic->statisticable_uuid)
      ->where('user_uuid', $statistic->user_uuid)->exists();
        $contentStatistic = ContentStatistic::where('uuid', $statistic->statisticable_uuid)
      ->where('type', $contentType)
      ->whereDate('date', $date->toDateString())->first();

        if ($contentStatistic === null) {
            $likes = $comments = 0;
            if ($withLikes) {
                $likes = Reaction::where('reactionable_uuid', $statistic->statisticable_uuid)
          ->where('type', 'like')
          ->where('reactionable_type', $statistic->statisticable_type)
          ->whereDate('created_at', $date->toDateString())->count();
            }

            if ($withComments) {
                if (! $this->comments->has($statistic->statisticable_uuid)) {
                    $this->comments[$statistic->statisticable_uuid] = DB::table('inside_pivots')
            ->where('related_type', Comments::class)->where(
                'parent_uuid',
                $statistic->statisticable_uuid
            )->where(
                'related_field',
                'comments'
            )->get();
                }
                foreach ($this->comments[$statistic->statisticable_uuid] as $pivot) {
                    $content = call_user_func($pivot->parent_type.'::find', $pivot->parent_uuid);
                    if (
                        ! is_null($content) &&
                        get_date($content->created_at)?->isSameDay($date)
                    ) {
                        $comments++;
                    }
                }
            }

            ContentStatistic::create(
                [
                    'title' => $statistic->statisticable->title,
                    'uuid' => $statistic->statisticable_uuid,
                    'type' => $contentType,
                    'status' => $statistic->statisticable->status,
                    'slug' => $statistic->statisticable->slug[0] ?? null,
                    'category_uuid' => $category->uuid ?? null,
                    'category_title' => $category->title ?? null,
                    'langcode' => $statistic->statisticable->langcode,
                    'visits' => 1,
                    'unique_visits' => $userHasVisited ? 0 : 1,
                    'likes' => $likes,
                    'comments' => $comments,
                    'created_at' => get_date($statistic->statisticable->created_at),
                    'updated_at' => get_date($statistic->statisticable->updated_at),
                    'published_at' => $statistic->statisticable->published_at,
                    'date' => $date,
                ]
            );
        } else {
            $contentStatistic->increment('visits');
            if (! $userHasVisited) {
                $contentStatistic->increment('unique_visits');
            }
        }
        if (! $userHasVisited) {
            ContentVisit::create([
                'user_uuid' => $statistic->user_uuid,
                'uuid' => $statistic->statisticable_uuid,
                'type' => $contentType,
            ]);
        }
    }
}
