<?php

namespace Inside\Statistics\Jobs;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Inside\Authentication\Models\User;
use Inside\Jobs\Job;
use Inside\Statistics\Facades\Stats;
use Inside\Statistics\Models\AuthenticationStatistic;
use Inside\Statistics\Models\Statistic;
use Symfony\Component\Console\Output\OutputInterface;

class ComputeAuthenticationStatistics extends Job
{
    protected int $chunk = 1000;

    public int $timeout = 14400;

    /**
     * @var string|null
     */
    public $queue;

    public function __construct(
    protected ?Carbon $from = null,
    protected ?Carbon $to = null,
    protected bool $append = false,
    protected ?OutputInterface $output = null
  ) {
        $this->queue = get_low_priority_queue_name();
    }

    protected function cleanup(): void
    {
        if (! $this->append) {
            AuthenticationStatistic::truncate();
        } else {
            AuthenticationStatistic::query()
              ->when($this->from !== null, fn ($query) => $query->whereDate('date', '>=', $this->from->toDateString()))
              ->when($this->to !== null, fn ($query) => $query->whereDate('date', '<=', $this->to->toDateString()))
              ->delete();
        }
    }

    /**
     * @return void
     */
    public function handle(): void
    {
        $this->cleanup();
        $total = Statistic::where('type', 'authentication')->when(
            $this->from !== null,
            function ($query) {
                return $query->whereDate('created_at', '>=', $this->from->toDateString());
            }
        )->when(
            $this->to !== null,
            function ($query) {
                return $query->whereDate('created_at', '<=', $this->to->toDateString());
            }
        )->count();

        $bar = null;
        if ($this->output) {
            $bar = $this->output->createProgressBar($total);
            $bar->start();
        }

        $authStats = collect();
        Statistic::where('type', 'authentication')->when(
            $this->from !== null,
            function ($query) {
                return $query->whereDate('created_at', '>=', $this->from->toDateString());
            }
        )->when(
            $this->to !== null,
            function ($query) {
                return $query->whereDate('created_at', '<=', $this->to->toDateString());
            }
        )->each(
            function (Statistic $statistic) use ($bar, $authStats) {
                if ($bar) {
                    $bar->advance();
                }
                if ($statistic->statisticable === null) {
                    return;
                }
                $contentType = class_to_type($statistic->statisticable_type);
                if ($contentType !== 'users') {
                    return;
                }
                $user = User::find($statistic->statisticable_uuid);
                if ($user === null || $user->information === null || $user->information->is_maintenance) {
                    return;
                }
                $date = get_date((int) $statistic->created_at)->setTime(12, 0);
                $key = md5($date->format('Ymd').'_'.$user->uuid);

                if (
                    ($authenticationStatistic = AuthenticationStatistic::where('uuid', $user->uuid)
                      ->whereDate(
                          'date',
                          $date->toDateString()
                      )->first()) === null
                ) {
                    if (! $authStats->has($key)) {
                        $authStats[$key] = [
                            'uuid' => $user->uuid,
                            'full_name' => Stats::getFullName($user->information),
                            'email' => $user->email,
                            'enabled' => $user->status,
                            'roles' => $user->permission->roles->pluck('name')->toJson(),
                            'connection_count' => 1,
                            'last_login_at_timestamp' => $user->last_login_at,
                            'date' => $date,
                        ];
                    } else {
                        $stat = $authStats[$key];
                        $stat['connection_count']++;
                        $authStats[$key] = $stat;
                    }
                } else {
                    $authenticationStatistic->increment('connection_count');
                }
            }
        );
        if ($bar) {
            $bar->finish();
        }

        if ($this->output) {
            $bar = $this->output->createProgressBar($authStats->count());
            $bar->start();
        }
        foreach ($authStats->chunk($this->chunk) as $chunked) {
            DB::table('authentication_statistics')->insert($chunked->toArray());
            if ($bar) {
                $bar->advance($chunked->count());
            }
        }

        if ($bar) {
            $bar->finish();
            $this->output->writeln('');
            $this->output->writeln('<fg=cyan>'.str_repeat('#', 80).'</>');
        }
    }
}
