<?php

namespace Inside\Statistics\Http\Controllers\AdvancedStatistics;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Inside\Content\Facades\Schema as InsideSchema;
use Inside\Permission\Facades\Role;
use Inside\Statistics\Services\AdvancedStatisticsService;

class AdvancedStatistics
{
    public function __construct(private AdvancedStatisticsService $advancedStatisticsService)
    {
    }

    public function getRoleCommitmentStatistics(Request $request)
    {
        return $this->handleStatisticsRequest($request, function ($monthId, $langcode, $contentTypeClass) {
            return $this->advancedStatisticsService->getRoleCommitmentStatistics($monthId, $contentTypeClass);
        });
    }

    public function getTopViewedContents(Request $request)
    {
        return $this->handleStatisticsRequest($request, function ($monthId, $langcode, $contentTypeClass) {
            return $this->advancedStatisticsService->getTopViewedContent($monthId, $langcode, $contentTypeClass);
        });
    }

    public function getTopLikedContents(Request $request)
    {
        return $this->handleStatisticsRequest($request, function ($monthId, $langcode, $contentTypeClass) {
            return $this->advancedStatisticsService->getTopLikedContent($monthId, $langcode, $contentTypeClass);
        });
    }

    public function getTopViewedCategories(Request $request)
    {
        return $this->handleStatisticsRequest($request, function ($monthId, $langcode, $contentTypeClass) {
            return $this->advancedStatisticsService->getTopCategories($monthId, $langcode, $contentTypeClass);
        });
    }

    public function getTopCommentedContent(Request $request)
    {
        return $this->handleStatisticsRequest($request, function ($monthId, $langcode, $contentTypeClass) {
            return $this->advancedStatisticsService->getTopCommentedContent($monthId, $langcode, $contentTypeClass);
        });
    }

    public function getBottomContents(Request $request)
    {
        return $this->handleStatisticsRequest($request, function ($monthId, $langcode, $contentTypeClass) {
            return $this->advancedStatisticsService->getBottomContent($monthId, $langcode, $contentTypeClass);
        });
    }

    private function handleStatisticsRequest(Request $request, callable $callback)
    {
        $filters = json_decode($request->get('filters', '[]'), true);
        $langcode = $filters['langcode'] ?? config('app.locale');
        $monthId = (int) $request->input('month_id');
        $contentTypeClass = type_to_class($request->input('content_type'));

        $result = $callback($monthId, $langcode, $contentTypeClass);

        return response()->json($result);
    }

    public function getMonths(Request $request)
    {
        $filters = json_decode($request->get('filters', []), true);
        $langcode = $filters['langcode'] ?? config('app.locale');
        $months = DB::table('inside_advanced_stats_months')
            ->orderBy('year', 'desc')
            ->orderBy('month', 'desc')
            ->get()
            ->map(function ($item) use ($langcode) {
                /** @var Carbon $date */
                $date = Carbon::createFromDate($item->year, $item->month, 1)
                    ->locale($langcode);

                $formatted = $date->translatedFormat('F Y');

                return [
                    'value' => $item->id,
                    'label' => ucfirst($formatted),
                ];
            });

        return response()->json($months);
    }

    public function getYears(Request $request)
    {
        return DB::table('inside_advanced_stats_months')
            ->select('year')
            ->distinct()
            ->orderBy('year', 'desc')
            ->pluck('year')
            ->map(fn ($year) => [
                'label' => $year,
                'value' => $year,
            ])
            ->values()
            ->toArray();
    }

    public function getMonthlyAttendanceStatsByYear(Request $request): array
    {
        $year = (int) $request->get('year', Carbon::now()->year);

        return DB::table('inside_advanced_stats_attendance as a')
            ->join(
                'inside_advanced_stats_months as m',
                'm.id',
                '=',
                'a.month_id'
            )
            ->where('m.year', $year)
            ->orderBy('m.month')
            ->get([
                'm.month',
                'a.authenticated_nbr',
                'a.authenticated_percentage',
                'a.access_nbr',
                'a.access_percentage',
            ])
            ->map(function ($row) {
                return [
                    'month' => (int) $row->month,
                    'authenticated' => [
                        'nbr'        => (int) $row->authenticated_nbr,
                        'percentage' => (float) $row->authenticated_percentage,
                    ],
                    'access' => [
                        'nbr'        => (int) $row->access_nbr,
                        'percentage' => (float) $row->access_percentage,
                    ],
                ];
            })
            ->toArray();
    }

    public function getRoleAccessPerMonth()
    {
        $rows = DB::table('inside_advanced_stats_activity_by_role as a')
            ->join('inside_advanced_stats_months as m', 'm.id', '=', 'a.month_id')
            ->select(
                'm.year',
                'm.month',
                'a.role_name',
                'a.access_percentage'
            )
            ->orderBy('m.year')
            ->orderBy('m.month')
            ->orderBy('a.role_name')
            ->get();

        $result = [];

        foreach ($rows as $row) {
            $monthKey = $row->month;

            if (! isset($result[$monthKey])) {
                $result[$monthKey] = [];
            }

            $result[$monthKey][] = [
                'role_name' => $row->role_name,
                'access_percentage' => (float) $row->access_percentage,
            ];
        }

        return response()->json($result);
    }

    public function getYearlyGeneralStats(Request $request)
    {
        $filters = json_decode($request->get('filters', '[]'), true);
        $langcode = $filters['langcode'] ?? config('app.locale');

        $year = (int) $request->get('year', Carbon::now()->year);

        $months = DB::table('inside_advanced_stats_months')
            ->where('year', $year)
            ->orderBy('month')
            ->get(['id', 'month']);

        $generalStats = DB::table('inside_advanced_stats_general')
            ->where('langcode', $langcode)
            ->whereIn('month_id', $months->pluck('id'))
            ->get(['month_id', 'data']);

        $generalStatsByMonth = $months->mapWithKeys(function ($month) use ($generalStats) {
            $stats = $generalStats->firstWhere('month_id', $month->id);
            $data = $stats ? json_decode($stats->data, true) : [];
            $collection = collect($data['activity'])->keyBy('type');

            $result = [
                'Likes' => $collection->get('likes')['value'] ?? 0,
                'Comments' => $collection->get('comments')['value'] ?? 0,
                'Notifications' => ($collection->get('notifications_web')['value'] ?? 0) +
                    ($collection->get('notifications_email')['value'] ?? 0),
            ];

            return [
                $month->month => $result,
            ];
        });

        return response()->json($generalStatsByMonth);
    }

    public function getStatistics(Request $request)
    {
        $monthId = (int) $request->input('month_id');
        $filters = json_decode($request->get('filters', []), true);
        $langcode = $filters['langcode'] ?? config('app.locale');

        $monthExists = DB::table('inside_advanced_stats_months')
            ->where('id', $monthId)
            ->exists();

        if (! $monthExists) {
            return response()->json([
                'attendance'        => [],
                'activity'          => [],
                'contentStatistics' => [],
            ]);
        }

        $activityRows = DB::table('inside_advanced_stats_activity_by_role')
            ->where('month_id', $monthId)
            ->orderBy('users_nbr', 'desc')
            ->get();

        $activity = $activityRows->map(function ($row) {
            return [
                'label' => ucfirst(Role::getHumanName($row->role_name))." ({$row->users_nbr})",
                'value' => (float) $row->access_percentage,
            ];
        })->values();

        $classTypeFilter = $request->get('class_type');

        $generalStats = DB::table('inside_advanced_stats_general')
            ->where('langcode', $langcode)
            ->where('month_id', $monthId)
            ->first();
        /** @phpstan-ignore-next-line */
        $generalStatsData = json_decode($generalStats->data, true);
        $administration = array_map(function ($item) {
            $schema = InsideSchema::getSchemaInformation($item['content_type']) ?? [];

            $item['label'] = $schema['options']['title'] ?? null;

            return $item;
        }, $generalStatsData['administration'] ?? []);

        return response()->json([
            'attendance' => $this->advancedStatisticsService->getAttendenceStatsDisplay($monthId),
            'activity' => $activity,
            'roleCommitmentStatistics' => $this->advancedStatisticsService->getRoleCommitmentStatistics($monthId, $classTypeFilter),
            'contentStatistics' => [
                'activity' => $generalStatsData['activity'] ?? [],
                'topViewedContent' => $this->advancedStatisticsService->getTopViewedContent($monthId, $langcode, $classTypeFilter),
                'topCategories' => $this->advancedStatisticsService->getTopCategories($monthId, $langcode, $classTypeFilter),
                'bottomContent' => $this->advancedStatisticsService->getBottomContent($monthId, $langcode, $classTypeFilter),
                'topLikedContent' => $this->advancedStatisticsService->getTopLikedContent($monthId, $langcode, $classTypeFilter),
                'topCommentedContent' => $this->advancedStatisticsService->getTopCommentedContent($monthId, $langcode, $classTypeFilter),
            ],
            'administration' => $administration,
        ]);
    }
}
