<?php

namespace Inside\Statistics\Exports;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Inside\Content\Facades\ContentHelper;
use Inside\Database\Eloquent\Builder;
use Inside\Statistics\Repositories\StatisticsBetaRepository;
use Maatwebsite\Excel\Concerns\RegistersEventListeners;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;

class UserVisitStatsExport extends AbstractStatsExport implements WithStrictNullComparison, WithEvents
{
    use RegistersEventListeners;

    protected array $config;

    protected StatisticsBetaRepository $repository;

    public function __construct(
        protected string $contentType,
        protected array $filters,
        protected ?string $search = null
    ) {
        parent::__construct('user_visits');

        $this->config = config('statistics.types.'.$this->contentType, []);
        $this->repository = new StatisticsBetaRepository();
    }

    /**
     * @return int
     */
    public function querySize(): int
    {
        $query = clone $this->query();
        $countQuery = "select count(*) as aggregate from ({$query->toSql()}) c";

        return Arr::first(DB::select($countQuery, $query->getBindings()))->aggregate;
    }

    /**
     * @return Builder
     */
    public function query(): Builder
    {
        $query = $this->repository->getUserVisitStatsQuery($this->contentType, $this->filters, $this->search);
        ContentHelper::applySortToQuery(
            $query,
            $this->filters,
            'date',
            'desc',
            [
                'full_name',
                'email',
                'visits',
                'unique_visits',
            ]
        );

        return $query;
    }

    /**
     * @param mixed $row
     * @return array
     */
    public function map($row): array
    {
        $mapped = [
            $row->full_name,
            $row->email,
        ];
        $mapped = array_merge($mapped, parent::map($row));

        return array_merge(
            $mapped,
            [
                $row->total_views,
                $row->unique_views,
            ]
        );
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        $headings = [
            Lang::get('statistics.user_visit.headings.full_name'),
            Lang::get('statistics.user_visit.headings.email'),
        ];
        $headings = array_merge($headings, parent::headings());

        return array_merge(
            $headings,
            [
                Lang::get('statistics.user_visit.headings.visits'),
                Lang::get('statistics.user_visit.headings.unique_visits'),
            ]
        );
    }

    /**
     * @return array
     */
    public function getCsvSettings(): array
    {
        return [
            'delimiter' => ';',
            'enclosure' => '"',
            'line_ending' => PHP_EOL,
            'use_bom' => true,
            'include_separator_line' => false,
            'excel_compatibility' => false,
        ];
    }
}
