<?php

namespace Inside\Statistics\Exports;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Facades\Schema;
use Inside\Database\Eloquent\Builder;
use Inside\Statistics\Repositories\StatisticsBetaRepository;
use Inside\Statistics\Repositories\StatisticsRepository;
use Inside\Support\Str;
use Maatwebsite\Excel\Concerns\RegistersEventListeners;
use Maatwebsite\Excel\Concerns\WithColumnFormatting;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Concerns\WithStrictNullComparison;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Cell\Hyperlink;
use PhpOffice\PhpSpreadsheet\Shared\Date;
use PhpOffice\PhpSpreadsheet\Style\NumberFormat;

class ContentStatsExport extends AbstractStatsExport implements WithColumnFormatting, WithStrictNullComparison, WithEvents
{
    use RegistersEventListeners;

    protected StatisticsBetaRepository $repository;

    protected array $config;

    public function __construct(
        protected string $contentType,
        protected array $filters
    ) {
        parent::__construct('content');

        $this->config = config('statistics.types.'.$this->contentType, []);
        $this->repository = new StatisticsBetaRepository();
    }

    public static function afterSheet(AfterSheet $event): void
    {
        foreach ($event->sheet->getColumnIterator($event->sheet->getHighestColumn()) as $row) {
            foreach ($row->getCellIterator() as $cell) {
                if (Str::contains($cell->getValue(), '://')) {
                    $cell->setHyperlink(new Hyperlink($cell->getValue(), 'Edit'));
                    $event->sheet->getStyle($cell->getCoordinate())->applyFromArray(
                        [
                            'font' => [
                                'color' => ['rgb' => '0000FF'],
                                'underline' => 'single',
                            ],
                        ]
                    );
                }
            }
        }
    }

    public function querySize(): int
    {
        $query = clone $this->query();
        $countQuery = "select count(*) as aggregate from ({$query->toSql()}) c";

        return Arr::first(DB::select($countQuery, $query->getBindings()))->aggregate;
    }

    public function query(): Builder
    {
        $query = $this->repository->getContentStatsQuery($this->contentType, $this->filters);
        ContentHelper::applySortToQuery(
            $query,
            $this->filters,
            'published_at',
            'desc',
            [
                'title',
            ]
        );

        return $query;
    }

    public function map($row): array
    {
        if ($row->archived) {
            $status = __('statistics.export.status.archived');
        } elseif ($row->status) {
            $status = __('statistics.export.status.true');
        } else {
            $status = __('statistics.export.status.false');
        }

        $mapped = [
            $row->content->title,
            Date::dateTimeToExcel(get_date($row->content->created_at)?->toDateTime()),
            Date::dateTimeToExcel(get_date($row->content->updated_at)?->toDateTime()),
            Date::dateTimeToExcel(get_date($row->content->published_at)?->toDateTime()),
            $status,
        ];
        $categoryField = isset($this->config['category']) ? Str::camel($this->config['category']) : null;
        if ($categoryField) {
            $mapped[] = $row->content->{$categoryField}->pluck('title')->implode(' & '); // We still trust config !
        }
        $mapped[] = $row->total_views;
        $mapped[] = $row->unique_views;

        if (isset($this->config['likes']) && $this->config['likes']) {
            $mapped[] = $row->likes;
        }

        if (isset($this->config['comments']) && $this->config['comments']) {
            $mapped[] = $row->content->comments->count();
        }
        $mapped[] = url('edit/'.$this->contentType.'/'.$row->content->uuid);

        return $mapped;
    }

    public function headings(): array
    {
        $headings = [
            Lang::get('statistics.content.headings.title'),
            Lang::get('statistics.content.headings.created_at'),
            Lang::get('statistics.content.headings.updated_at'),
            Lang::get('statistics.content.headings.published_at'),
            Lang::get('statistics.content.headings.status'),
        ];

        if (isset($this->config['category'])) {
            try {
                $fieldOptions = Schema::getFieldOptions($this->contentType, $this->config['category']);
            } catch (Exception $exception) {
                Log::error('[ContentStatsExport::headings] '.$exception->getMessage());
            }
            $headings[] = $fieldOptions['title'][App::getLocale()] ?? '';
        }
        $headings[] = Lang::get('statistics.content.headings.visits');
        $headings[] = Lang::get('statistics.content.headings.unique_visits');

        if (isset($this->config['likes']) && $this->config['likes']) {
            $headings[] = Lang::get('statistics.content.headings.likes');
        }

        if (isset($this->config['comments']) && $this->config['comments']) {
            $headings[] = Lang::get('statistics.content.headings.comments');
        }
        $headings[] = Lang::get('statistics.content.headings.link');

        return $headings;
    }

    public function getCsvSettings(): array
    {
        return [
            'delimiter' => ';',
            'enclosure' => '"',
            'line_ending' => PHP_EOL,
            'use_bom' => true,
            'include_separator_line' => false,
            'excel_compatibility' => false,
        ];
    }

    /**
     * @return array
     */
    public function columnFormats(): array
    {
        return [
            'B' => NumberFormat::FORMAT_DATE_DDMMYYYY,
            'C' => NumberFormat::FORMAT_DATE_DDMMYYYY,
            'D' => NumberFormat::FORMAT_DATE_DDMMYYYY,
        ];
    }
}
