<?php

namespace Inside\Statistics\Exports;

use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Database\Eloquent\Collection as DatabaseCollection;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Str;
use Inside\Content\Models\Model;
use Maatwebsite\Excel\Concerns\Exportable;
use Maatwebsite\Excel\Concerns\FromQuery;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;
use Maatwebsite\Excel\Concerns\WithCustomCsvSettings;
use Maatwebsite\Excel\Concerns\WithCustomQuerySize;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;

abstract class AbstractStatsExport implements FromQuery, WithMapping, ShouldQueue, ShouldAutoSize, WithCustomCsvSettings, WithHeadings, WithCustomQuerySize
{
    use Exportable;

    public function __construct(
        protected string $type
    ) {
    }

    public function map($row): array
    {
        $mapped = [];
        if (! is_null($contentType = $row->getOriginal('content_type')) && $contentType != $row->content_type) {
            $correctModelClass = type_to_class($contentType);
            $row = new $correctModelClass($row->getOriginal());
        }

        foreach (array_keys(config('statistics.export.'.$this->type.'.headers', [])) as $key) {
            $value = '';
            $theModel = $row;
            if (Str::contains((string) $key, '.')) {
                [$subModel, $key] = explode('.', (string) $key, 2);
                if ($subModel === 'statisticable') {
                    $theModel = $row->statisticable;
                } elseif ($subModel === 'user') {
                    $theModel = $row->user;
                }
            }
            if (isset($theModel->{$key})) {
                if (is_string($theModel->{$key})) {
                    $value = match ($key) {
                        'full_name' => $theModel->status === false ? trim(
                            __('anonymous.user.firstname').' '.__(
                                'anonymous.user.lastname'
                            )
                        ) : $theModel->{$key},
                        'email'     => $theModel->status === false ? '_' : $theModel->{$key},
                        default     => $theModel->{$key}
                    };
                } elseif ($theModel->{$key} instanceof Collection) {
                    $value = implode(', ', $theModel->{$key}->pluck('title')->toArray());
                }
            }
            $callback = config('statistics.export.'.$this->type.'.presenters.'.$key);
            if (! is_null($callback) && is_callable($callback)) {
                $value = $callback($value, $row);
            }
            $mapped[] = $value;
        }

        return $mapped;
    }

    public function headings(): array
    {
        $headings = [];
        foreach (config('statistics.export.'.$this->type.'.headers', []) as $key => $defaultLabel) {
            if (Lang::get('statistics.'.$this->type.'.headings.'.$key) === 'statistics.'.$this->type
                .'.headings.'.$key
            ) {
                $headings[] = $defaultLabel;
            } else {
                $headings[] = Lang::get('statistics.'.$this->type.'.headings.'.$key);
            }
        }

        return $headings;
    }

    public function querySize(): int
    {
        $query = clone $this->query();
        $countQuery = "select count(*) as aggregate from ({$query->toSql()}) c";

        return Arr::first(DB::select($countQuery, $query->getBindings()))->aggregate;
    }

    public function getCsvSettings(): array
    {
        return [
            'delimiter' => ',',
            'enclosure' => '"',
            'line_ending' => PHP_EOL,
            'use_bom' => true,
            'include_separator_line' => false,
            'excel_compatibility' => false,
        ];
    }
}
