<?php

namespace Inside\Statistics\Console;

use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Inside\Console\Command;
use Inside\Content\Facades\Schema as InsideSchema;

class CreateStatsTables extends Command
{
    /**
     * @var string
     */
    protected $signature = 'inside:stats:tables';

    /**
     * @var string
     */
    protected $description = 'Read stats config and create required tables';

    /**
     * @return void
     */
    public function handle(): void
    {
        $this->info('Creating stats tables started...');
        $statsTypes = config('statistics.types', []);
        if (! $statsTypes) {
            $this->error('Can\'t create stats tables there is no types configured !');

            return;
        }

        collect($statsTypes)->keys()->each(function ($type) {
            $this->createStatsTable($type);
        });

        // Table for authentication statistics
        if (Schema::hasTable('inside_authentication_statistics')) {
            $this->warning('Table : inside_authentication_statistics already exists !');

            return;
        }

        Schema::create('inside_authentication_statistics', function (Blueprint $table) {
            $table->uuid('user_uuid');
            $table->date('created_at');
            $table->integer('connections')->default(0);
            $table->primary(['user_uuid', 'created_at']);
            $table->foreign('user_uuid')->references('uuid')->on('inside_users')->onDelete('cascade');
        });
    }

    private function createStatsTable(string $type): void
    {
        if (! InsideSchema::hasContentType($type)) {
            $this->error('Content type : '.$type.' does not exist !');

            return;
        }
        $statsTableName = type_to_stats_table($type);
        if (Schema::hasTable($statsTableName)) {
            $this->warning('Table :'.$statsTableName.' already exists !');

            return;
        }
        try {
            Schema::create($statsTableName, function (Blueprint $table) use ($type) {
                $table->bigIncrements('id');
                $table->uuid('user_uuid');
                $table->uuid('content_uuid');
                $table->foreign('user_uuid')->references('uuid')->on('inside_content_users')->onDelete('cascade');
                $table->foreign('content_uuid')->references('uuid')->on(type_to_table($type))->onDelete('cascade');
                $table->date('date');
            });
            $this->info('Stats table '.$statsTableName.' of type : '.$type.' has been created !');
        } catch (\Exception $e) {
            Schema::dropIfExists($statsTableName);
            $this->error('Error occurred while trying to create state table : '.$e->getMessage());
        }
    }
}
