<?php

namespace Inside\Slug\Services;

use Cocur\Slugify\Slugify;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Slug\Models\Slug;

class SlugService
{
    protected Model $model;

    /**
     * @throws Exception
     */
    public function slug(Model $model): void
    {
        $this->model = $model;
        if (! $this->model->uuid) {
            return;
        }

        try {
            $options = Schema::getModelOptions(class_to_type($model));
        } catch (ModelSchemaNotFoundException $exception) {
            return;
        }

        if (! isset($options['aliasable']) || ! $options['aliasable']) {
            return;
        }

        $this->deleteSlug($this->model);
        foreach ($this->model->sluggable() as $config) {
            $config = $this->getConfiguration($config);
            $source = $this->getSource($config['source']);
            $parents = $this->getParent($config['parent']);

            if ($source || is_numeric($source)) {
                $slug = $this->generate($source, $config);
                $slug = $this->makeUnique($slug, $config);
                $this->save($slug);
                $this->prefixParent($parents, $slug, $config);
            }
        }
    }

    public function prefixParent(array $parents, string $slug, array $config): void
    {
        foreach ($parents as $parent) {
            $parentSlug = $parent.'/'.$slug;
            $parentSlug = $this->makeUnique($parentSlug, $config);
            $this->save($parentSlug);
        }
    }

    public function getParent(?array $from = []): array
    {
        if (empty($from)) {
            return [];
        }

        $sources = [];
        foreach ($from as $key) {
            if (method_exists($this->model, $key)) {
                $parents = $this->model->{$key}()->get();
                if ($parents->count()) {
                    foreach ($parents as $parent) {
                        $sources = array_merge($sources, $parent->slug);
                    }
                }
            }
        }

        return $sources;
    }

    public function getConfiguration(array $overrides = []): array
    {
        $defaultConfig = config('sluggable', []);

        return array_merge($defaultConfig, $overrides);
    }

    public function getUuid(string $type, string $slug): ?string
    {
        $result = Slug::whereSlug($slug)->whereType($type)->first();

        return $result?->uuid;
    }

    public function getSource(mixed $from = null): ?string
    {
        if (is_null($from)) {
            return $this->model->__toString();
        }

        $source = array_map(
            function ($key) {
                $value = data_get($this->model, $key);
                if (is_bool($value)) {
                    $value = (int) $value;
                }

                return $value;
            },
            (array) $from
        );

        return implode(' ', $source);
    }

    public function save(string $slug): void
    {
        $data = [
            'uuid' => $this->model->uuid,
            'type' => $this->model->getTable(),
            'langcode' => $this->model->langcode,
            'slug' => $slug,
        ];
        $slug = new Slug();
        $slug->fill($data)->save();
    }

    /**
     * @throws Exception
     */
    public function deleteSlug(Model $model): void
    {
        Slug::where('uuid', '=', $model->uuid)->where('langcode', '=', $model->langcode)->delete();
    }

    protected function generate(string $source, array $config): string
    {
        $engine = new Slugify();
        $slug = $engine->slugify($source, $config['separator']);
        $length = mb_strlen($slug);

        if (is_string($slug) && $length > $config['maxLength']) {
            $reverseOffset = $config['maxLength'] - $length;
            $lastSeparatorPos = mb_strrpos($slug, $config['separator'], $reverseOffset);

            if ($config['maxLengthKeepWords'] && $lastSeparatorPos !== false) {
                return mb_substr($slug, 0, $lastSeparatorPos);
            }

            return trim(mb_substr($slug, 0, $config['maxLength']), $config['separator']);
        }

        return $slug;
    }

    protected function makeUnique(string $slug, array $config): string
    {
        $list = $this->getExistingSlugs($slug, $config);

        if ($list->count() === 0 || $list->contains($slug) === false) {
            return $slug;
        }

        $suffix = $this->generateSuffix($slug, $config, $list);

        return $slug.$config['separator'].$suffix;
    }

    protected function generateSuffix(string $slug, array $config, Collection $list): string
    {
        for ($suffix = 1; ($suffix < $list->count()) &&
        $list->contains($slug.$config['separator'].$suffix); $suffix++) {
        }

        return (string) $suffix;
    }

    /**
     * Get all existing slugs that are similar to the given slug.
     */
    protected function getExistingSlugs(string $slug, array $config): Collection
    {
        $query = Slug::where(fn ($query) => $query->where('slug', '=', $slug)->orWhere('slug', 'LIKE', "$slug{$config['separator']}%"));

        if (! config('slug.unique')) {
            $query
                ->where('type', '=', $this->model->getTable())
                ->where('langcode', '=', $this->model->langcode);
        }

        return $query->get(['slug'])->pluck('slug');
    }

    public function setModel(Model $model): static
    {
        $this->model = $model;

        return $this;
    }

    public function getParentSlug(string $type, string $id): ?string
    {
        $content = call_user_func(type_to_class($type).'::find', $id);

        if (! $content) {
            return null;
        }

        // do some logic here when we have clean URls
        $url = '';

        $customCallback = config('slug.custom_parent_slug', false);
        if ($customCallback && is_callable($customCallback)) {
            $customCallback($content, $url);
        }

        return $url;
    }
}
