<?php

namespace Inside\Sharepoint\Services;

use Inside\Facades\Http;

class SharePointService
{
    private string $token;
    private string $graphBase;

    public function __construct()
    {
        $this->graphBase = config('sharepoint.graph.base_url');
        $this->token = $this->authenticate();
    }

    private function authenticate(): string
    {
        $response = Http::asForm()->post(
            "https://login.microsoftonline.com/" .
            config('sharepoint.tenant_id') .
            "/oauth2/v2.0/token",
            [
                'client_id'     => config('sharepoint.client.id'),
                'client_secret' => config('sharepoint.client.secret'),
                'scope'         => config('sharepoint.graph.scope'),
                'grant_type'    => 'client_credentials',
            ]
        );

        return $response->json()['access_token'];
    }

    public function listItems(string $driveId, ?string $itemId = null): array
    {
        $endpoint = $itemId
            ? "{$this->graphBase}/drives/{$driveId}/items/{$itemId}/children"
            : "{$this->graphBase}/drives/{$driveId}/root/children";

        $currentItem = [];
        $parentItem = [];
        $childrenItems = Http::withToken($this->token)
            ->get($endpoint)
            ->json();

        if($itemId) {
            $currentItem = Http::withToken($this->token)
                ->get("{$this->graphBase}/drives/{$driveId}/items/{$itemId}")
                ->json();
            $parentItem = $currentItem['parentReference'] ?? [];
            
            if($currentItem['name'] == 'root') {
                $parentItem = [];
                $currentItem = [];
            }
            if(str($parentItem['path'])->endsWith('/root:')) {
                $parentItem = [];
            }
        }

        $response = collect($childrenItems['value'] ?? [])
            ->map(fn ($item) => $this->mapItem($item, $driveId))
            ->values()
            ->toArray();

        return [
            'items' => $response,
            'currentFolderInfo' => $itemId ? array_only($currentItem, ['id', 'name']) : [],
            'parentFolderInfo' => array_only($parentItem, ['id', 'name'])
        ];
    }

    private function mapItem(array $item, string $driveId): array
    {
        $isFolder = isset($item['folder']);

        return  [
            'id'   => $item['id'],
            'name' => $item['name'],
            'type' => $isFolder ? 'folder' : 'file',
            'size'        => $item['size'] ?? 0,
            'mime_type'   => $item['file']['mimeType'] ?? null,
            'modified_at' => $item['lastModifiedDateTime'] ?? null,
            'paths' => [
                'browse' => $isFolder
                    ? "/api/v1/sharepoint/browse/{$driveId}/{$item['id']}"
                    : null,

                'edit' => !$isFolder
                    ? $item['webUrl']
                    : null,
            ],
        ];
    }

    public function listDrives(string $siteId): array
    {
        $response = Http::withToken($this->token)
            ->get("{$this->graphBase}/sites/{$siteId}/drives")
            ->json();

        return collect($response['value'] ?? [])
            ->map(fn ($drive) => [
                'id'   => $drive['id'],
                'name' => $drive['name'],
                'type' => $drive['driveType'],
            ])
            ->values()
            ->toArray();
    }

    public function getSite(): array
    {
        $uri = sprintf(
            '%s/sites/%s:%s',
            $this->graphBase,
            config('sharepoint.site.host'),
            config('sharepoint.site.path')
        );

        return Http::withToken($this->token)
            ->get($uri)
            ->json();
    }
}
