<?php

namespace Inside\Settings\Services;

use Illuminate\Database\QueryException;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Inside\Authentication\Models\User;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Permission\Facades\Permission;
use Inside\Settings\Contracts\SettingService;
use Inside\Settings\Models\Setting;
use Inside\Support\Str;

/**
 * Class SettingStore
 *
 * Setting store service
 */
class SettingStore implements SettingService
{
    /**
     * saved config
     */
    protected array $data = [];

    protected array $hiddenSettings = [
        'gtag_general',
        'ga_general',
        'gtm_general',
    ];

    /**
     * Already loaded ?
     */
    protected bool $isLoaded = false;

    public function get(string $group, string $key, ?string $default = null): ?string
    {
        if ($default === null) {
            $default = config('settings.defaults.'.$key);
        }

        $this->load();

        return Arr::get($this->data, "$group.$key", $default);
    }

    public function has(string $group, string $key): bool
    {
        $this->load();

        return Arr::has($this->data, "$group.$key");
    }

    public function set(string $group, $key, $value = null): SettingService
    {
        $this->load();

        if (is_array($key)) {
            foreach ($key as $k => $v) {
                Arr::set($this->data, "$group.$k", $v);
                $this->save($group, $k, $v);
            }
        } else {
            Arr::set($this->data, "$group.$key", $value);
            $this->save($group, $key, $value);
        }

        return $this;
    }

    public function forget(string $group, string $key): SettingService
    {
        if ($this->has($group, $key)) {
            Arr::forget($this->data, $key);
            try {
                DB::table('inside_settings')->where('group', $group)->where('key', $key)->delete();
            } catch (QueryException $exception) {
                Log::error('[SettingStore::forget] fail to forget setting ['.
                    $group.']['.$key.'] => '.$exception->getMessage());
            }
        }

        return $this;
    }

    public function forgetAll(): SettingService
    {
        $this->data = [];
        DB::table('inside_settings')->truncate();

        return $this;
    }

    public function all(): array
    {
        $this->load();

        return $this->data;
    }

    protected function load(bool $force = false): void
    {
        if (! $this->isLoaded || $force) {
            $this->data = Setting::orderBy('group')->orderBy('key')->get()
                    ->groupBy('group')
                    ->transform(
                        function ($group) {
                            return $group->pluck('value', 'key');
                        }
                    )->toArray();
            $this->isLoaded = true;
        }
    }

    /**
     * Save a setting in database
     */
    private function save(string $group, string $key, mixed $value): void
    {
        Setting::updateOrCreate(
            [
                'group' => $group,
                'key' => $key,
            ],
            [
                'value' => $value,
            ]
        );
    }

    public function getExportableSettings(): Collection
    {
        $exportable = collect();
        $settings = Setting::where('group', 'not like', '\_%')
            ->whereNotIn('group', ['system', 'protected'])
            ->get();
        $assets = collect();
        foreach ($settings as $setting) {
            if (! $exportable->has($setting->group)) {
                $exportable[$setting->group] = collect();
            }
            if (in_array($setting->key, $this->hiddenSettings)) {
                continue;
            }
            $exportable[$setting->group][$setting->key] = $setting->value;
            if (Str::startsWith($setting->value, 'assets/settings/')) {
                if (Storage::disk('local')->exists($setting->value)) {
                    $assets[$setting->value] = base64_encode(Storage::disk('local')->get($setting->value));
                } else {
                    $exportable[$setting->group][$setting->key] = null;
                }
            }
        }

        return collect(['settings' => $exportable, 'assets' => $assets]);
    }

    public function importSettings(Collection $data): bool
    {
        if (! isset($data['settings']) || empty($data['settings'])) {
            return false;
        }
        if (! Storage::disk('local')->exists('assets/settings')) {
            Storage::disk('local')->makeDirectory('assets/settings');
        }
        $assets = $data['assets'];
        foreach ($data['settings'] as $group => $settings) {
            foreach ($settings as $key => $value) {
                setting($group, [$key => $value]);

                if (Str::startsWith($value, 'assets/settings/')) {
                    $asset = $assets->{$value};
                    if (isset($asset) && ! empty($asset)) {
                        Storage::disk('local')->put($value, base64_decode($asset));
                    }
                }
            }
        }

        return true;
    }

    protected function maskValue(string $value): string
    {
        $length = strlen($value);

        if ($length <= 12) {
            return str_repeat('*', $length);
        }

        $masked = substr($value, 0, 3).str_repeat('*', 15).substr($value, -3);

        return $masked;
    }

    protected function isMaskRequired(string $group, string $key): bool
    {
        $masking = config('secure_settings.masking', []);

        return isset($masking[$group]) && in_array($key, $masking[$group]);
    }

    public function isCurrentValueMasked(string $group, string $key, ?string $value): bool
    {
        if (is_null($value)) {
            return false;
        }

        if ($this->isMaskRequired($group, $key)) {
            return $value === $this->maskValue($value);
        }

        return false;
    }

    public function listForGroup(?string $group = null, bool $isMaeciaAdmin = false): Collection
    {
        $settings = null;

        if (! is_null($group)) {
            if (Str::startsWith('_', $group) && ! $isMaeciaAdmin) {
                throw new AuthorizationException();
            }

            $settings = Setting::where('group', $group)->orderBy('key')->pluck('value', 'key');
        } else {
            $settings = Setting::orderBy('group')->orderBy('key')
                ->when(! $isMaeciaAdmin,
                    function ($query) {
                        $query->where(function ($query) {
                            $query->where('group', 'not like', '\_%')->where('group', 'not like', '\#%');
                        });
                    }
                )->get()->groupBy('group')->transform(
                    function ($group) {
                        return $group->pluck('value', 'key');
                    }
                );
        }

        foreach (config('secure_settings.masking', []) as $group => $keys) {
            foreach ($keys as $key) {
                if ($value = data_get($settings, $group.'.'.$key)) {
                    if ($this->isMaskRequired($group, $key)) {
                        data_set($settings, $group.'.'.$key, $this->maskValue($value));
                    }
                }
            }
        }

        /** @var User|null $user */
        $user = Auth::user();
        if (! $user || Permission::backofficeAccessibleEntries($user)->isEmpty()) {
            $settings->forget(['ai', 'appearance', 'newsletter', 'system', '_admin']);
        }

        return $settings;
    }
}
