<?php

namespace Inside\Settings\Http\Controllers;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Inside\Permission\Exceptions\AuthorizationException;
use Inside\Settings\Services\FontValidatorService;
use Inside\Settings\Services\ImportFontsService;
use Laravel\Lumen\Routing\Controller;

class ImportFontsController extends Controller
{
    private const ALLOWED_PARAM_VALUES = [
        'font' => ['primary', 'secondary'],
        'type' => ['bold', 'normal', 'light'],
    ];

    public function __invoke(
        FontValidatorService $fontValidatorService,
        ImportFontsService $importFontsService,
        Request $request)
    : JsonResponse
    {
        // Check user authorization
        $me = Auth::user();
        if (! $me) {
            throw AuthorizationException::create('upload', 'settings');
        }

        $maxSize = UploadedFile::getMaxFilesize();
        $this->validate(
            $request,
            [
                'asset' => 'required|file|max:'.$maxSize,
                'font' => 'required|in:'.implode(',', self::ALLOWED_PARAM_VALUES['font']),
                'type' => 'required|in:'.implode(',', self::ALLOWED_PARAM_VALUES['type']),
            ]
        );

        $font = $request->get('font');
        $type = $request->get('type');

        /** @var UploadedFile $fontFile */
        $fontFile = $request->file('asset');
        $fontValidatorService->validate($fontFile);

        $data = $importFontsService->importFont($type, $font, $fontFile);

        return response()->json($data);
    }
}
