<?php

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Inside\Facades\Package;
use Inside\Search\Facades\Searchable;
use Inside\Support\Str;

if (! function_exists('search_namespace')) {
    /**
     * Get the search base namespace
     *
     * @param  string  $class
     *
     * @return string
     * @throws Exception
     */
    function search_namespace(string $class = ''): string
    {
        $driver = match (config('scout.driver')) {
            'mysql' => 'database',
            'pgsql' => 'database',
            'solr' => 'solr',
            default => throw new Exception('Driver not supported!')
        };

        if (! Package::has("inside-search-$driver")) {
            throw new Exception('No search module installed!');
        }

        return '\Inside\Search\\'.Str::studly($driver).($class ? '\\'.$class : $class);
    }
}

if (! function_exists('inside_search_force_reindex')) {
    /**
     * Force reindex to database
     *
     * WARNING : very consuming process
     * @throws Exception
     */
    function inside_search_force_reindex(Command $command = null): void
    {
        $searchableClasses = Searchable::getSearchableClasses();

        if ($command) {
            $bar = $command->getOutput()->createProgressBar($searchableClasses->count());
            $bar->setFormat("<fg=yellow;bg=blue>%message%</>\n %current%/%max% [%bar%] %percent:3s%%");
        }

        // Reindex everything
        foreach ($searchableClasses as $searchable) {
            if ($command) {
                $bar->setMessage($searchable->model);
            }

            $query = call_user_func('\\'.$searchable->model.'::query');

            if ($query) {
                $total = $query->count();
                $i = 0;
                while ($i < $total) {
                    $items = $query->skip($i)->take(config('scout.chunk.searchable'));
                    if ($items) {
                        $i += $items->count();
                        $items->searchable();
                    }
                }
            }

            if ($command) {
                $bar->advance();
            }
        }

        if ($command) {
            $bar->setMessage('DONE');
            $bar->finish();
        }
    }
}

if (! function_exists('inside_search_get_class_from_filter')) {
    /**
     * Get Model full class name from a filter
     *
     * @param  string  $filter  a filter ( index from filtering request )
     *
     * @return string
     */
    function inside_search_get_class_from_filter(string $filter): string
    {
        return '\\Inside\\Content\\Models\\Contents\\'.Str::studly($filter);
    }
}

if (! function_exists('inside_search_get_class_from_filters')) {
    /**
     * Get All Model full class name from a filter collection
     *
     * @param  Collection  $filters  filters ( index from filtering request )
     *
     * @return Collection
     * @throws Exception
     */
    function inside_search_get_class_from_filters(Collection $filters): Collection
    {
        $classes = collect();
        $searchableClasses = Searchable::getSearchableClasses();

        foreach ($filters as $filter) {
            if ($searchableClasses->has($filter)) {
                $classes[$filter] = $searchableClasses[$filter]->model;
            }
        }

        return $classes;
    }
}

if (! function_exists('inside_search_history_enabled')) {
    /**
     * check if history is enabled
     *
     * @return bool
     */
    function inside_search_history_enabled(): bool
    {
        return config('history.enabled');
    }
}
