<?php

namespace Inside\Search\Services;

use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection as EloquentDatabaseCollection;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Http\Request;
use Illuminate\Pagination\LengthAwarePaginator as CustomLengthAwarePaginator;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cookie;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Content\Helpers\SchemaHelper;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\Users;
use Inside\Content\Models\Model;
use Inside\Content\Transformers\ContentTransformer;
use Inside\Facades\Package;
use Inside\Search\Contracts\Search as SearchContract;
use Inside\Search\Facades\Searchable;
use Inside\Search\Models\SearchHistory;
use Laravel\Scout\Builder;
use Laravel\Scout\EngineManager;
use Ramsey\Uuid\Uuid;
use RuntimeException;

class Search implements SearchContract
{
    public const OPERATOR_OR = 0;

    public const OPERATOR_AND = 1;

    protected array $parameters = [];

    /**
     * The search term
     */
    protected string $search = '';

    /**
     * The search query
     */
    protected ?Builder $query = null;

    /**
     * The search filters
     */
    protected array $filters = [];

    /**
     * Wanted fields
     */
    protected array $fields = [];

    /**
     * Paginated result
     */
    protected int $limit = 0;

    /**
     * Content search result field
     */
    protected array $contentFields = [];

    protected array $postFilter = [];

    protected Request $request;

    /**
     * Build new search service instance
     */
    public function __construct(
    ?Request $request = null
  ) {
        if (is_null($request)) {
            /** @var Request $request */
            $request = request();
        }
        $this->request = $request;

        $this->parameters = $this->request->all();
    }

    public function withRequest(Request $request): self
    {
        $this->request = $request;

        return $this;
    }

    protected function prepareHistoryQueryForUser(): \Illuminate\Database\Eloquent\Builder
    {
        /** @var User $user */
        $user = Auth::user();

        return SearchHistory::where('user_uuid', $user->uuid)
      ->where('langcode', $this->filters['langcode']);
    }

    protected function getExistingSearch(): ?SearchHistory
    {
        return $this->prepareHistoryQueryForUser()->where('text', trim($this->search))->first(); // @phpstan-ignore-line
    }

    protected function removeExceedingSizeHistory(): void
    {
        $query = $this->prepareHistoryQueryForUser();
        $count = $query->count();
        if ($count >= config('history.size')) {
            $toRemove = $count === config('history.size') ? 1 : $count - config('history.size');
            $query->orderBy('last_searched_at')->take($toRemove)->delete();
        }
    }

    protected function addSearchToHistory(): void
    {
        /** @var User $user */
        $user = Auth::user();

        if (empty($this->search)) {
            return;
        }

        if ($existingHistory = $this->getExistingSearch()) {
            $existingHistory->last_searched_at = now();
            $existingHistory->save();
        } else {
            $this->removeExceedingSizeHistory();

            SearchHistory::create([
                'user_uuid' => $user->uuid,
                'langcode' => $this->filters['langcode'],
                'text' => trim($this->search),
            ]);
        }
    }

    /**
     * Execute search with parameters given in constructor
     *
     * @throws Exception
     */
    public function search(): LengthAwarePaginator | Collection
    {
        $this->setupSearch();
        if (is_null($this->query)) {
            throw new Exception('failed to init search query');
        }

        $this->setupFilters();

        $this->setupFields();

        $this->setupLanguage();

        $this->setupPaginate();

        $this->applyFilters();

        $this->applyCustomFilters();

        // Force not using Scout index but our system instead
        $this->query->within('*');

        if (inside_search_history_enabled()) {
            $this->addSearchToHistory();
        }

        if ($this->limit > 0) {
            $result = $this->query
                ->paginate($this->limit)
                ->appends($this->request->except('page'));

            return $this->rebuildPaginator($result);
        }

        return collect(['data' => $this->applyCustomTransformers($this->query->get())]);
    }

    /**
     * Clean search terms
     */
    public static function cleanTerms(string $data): string
    {
        $json = json_decode($data, true);
        if (! is_null($json) && isset($json['headers']) && isset($json['rows'])) {
            $results = [];
            foreach ($json['headers'] as $header) {
                $results[] = strip_tags($header['content']);
            }

            foreach ($json['rows'] as $row) {
                $line = [];
                foreach ($row['data'] as $cell) {
                    $results[] = strip_tags($cell['content']);
                }
            }

            if (! empty($results)) {
                return implode(' ', $results);
            }
        }

        // Remove unwanted stuffs
        $toBeRemoved = ['/&amp;/', '/&lt;/', '/&gt;/', '/&#039;/', '/&quot;/', '/&nbsp;/'];

        $data = htmlspecialchars_decode((string) preg_replace($toBeRemoved, '', $data));

        // Remove tags
        $data = strip_tags($data);

        // Remove dot comma ...
        $data = (string) preg_replace('/[’\n\r\s\"\'\-,�;:\/!\?@\._\n\r\s\)\(\t\�]+/u', ' ', $data);

        // Remove missing stuff
        return (string) preg_replace('/[^\p{L}\p{N}\s]/u', '', $data);
    }

    /**
     * Get the query term and init the query
     *
     * @throws Exception
     */
    protected function setupSearch(): void
    {
        if (isset($this->parameters['query'])) {
            $this->search = str($this->parameters['query'])->ascii();

            // Clean $search query in production
            if (App::environment(['production', 'staging'])) {
                $this->search = (string) preg_replace("/\*:\*/", '', $this->search);
                $this->search = (string) preg_replace("/^[^:]+:\*/", '', $this->search);
            }
        }

        $this->search = static::cleanTerms($this->search);

        // Init searcher with indexer(s)
        $this->query = call_user_func(search_namespace('Models\Index::search'), $this->search);
    }

    /**
     * Setup filters
     */
    protected function setupFilters(): void
    {
        if (isset($this->parameters['filters'])) {
            $this->filters = json_decode($this->parameters['filters'], true) ?? [];
        }

        if (isset($this->filters['in']) && is_string($this->filters['in'])) {
            // Note : Il traine des requêtes front avec un in:content_type au lieu de in:[content_type]
            $this->filters['in'] = [$this->filters['in']];
        }

        if (isset($this->filters['workflow']) && is_array($this->filters['workflow'])) {
            $this->filters['workflow_status:eq'] = (int) $this->filters['workflow']['status:eq'];
            unset($this->filters['workflow']);
        }

        foreach ($this->filters as $filter => $value) {
            if (
                strstr($filter, 'created_at')
                || strstr($filter, 'updated_at')
                || strstr($filter, 'published_at')
            ) {
                $name = $filter;
                $operator = null;
                if (strpos($filter, ':')) { // If front asked with operator
                    [$name, $operator] = explode(':', $filter, 2);
                }

                $this->filters[$name] = collect($this->filters[$name] ?? [])
                    ->push([
                        'operator' => $operator,
                        'value' => $value,
                    ])
                    ->all();

                unset($this->filters[$filter]);
            }
        }
    }

    /**
     * Setup fields
     */
    protected function setupFields(): void
    {
        if (isset($this->parameters['fields'])) {
            $this->fields = array_merge(
                ['url', 'score', 'content_type', 'type', 'published_at'],
                json_decode($this->parameters['fields'], true) ?? []
            );
        }
    }

    /**
     * Set up the pagination
     *
     * @throws Exception
     */
    protected function setupPaginate(): void
    {
        if (is_null($this->query)) {
            throw new Exception('Init search query not set');
        }

        // Front messed with param names alias it
        if (array_key_exists('sort', $this->filters)) {
            if (! array_key_exists('order', $this->filters)) {
                $this->filters['order'] = $this->filters['sort'];
            }
            unset($this->filters['sort']);
        }
        if (array_key_exists('offset', $this->filters)) {
            if (! array_key_exists('limit', $this->filters)) {
                $this->filters['limit'] = $this->filters['offset'];
            }
            unset($this->filters['offset']);
        }
        if (array_key_exists('paginate', $this->filters)) {
            unset($this->filters['paginate']);
        }

        // Prepare pagination
        $limit = array_key_exists(
            'limit',
            $this->filters
        ) ? (int) $this->filters['limit'] : null; // Null means no limit

        // Prepare order
        // example "order":"created_at:desc" default to created_at
        $order = $this->filters['order'] ?? ($this->filters['sort'] ?? 'score:desc');
        [$column, $direction] = strpos($order, ':') ? explode(':', $order, 2) : [$order, 'asc'];

        // Sur les annuaires, on force la recherche par score
        if (
            isset($this->filters['in']) &&
            count($this->filters['in']) == 1 &&
            in_array('users', $this->filters['in'])
        ) {
            $this->query->orderBy('score', 'desc');
        }

        if ($limit && $limit > 0) {
            // Result with pagination
            $this->limit = $limit;
        }
        $this->query->orderBy($column, $direction);
        if ($column != 'score') {
            // Always add score
            $this->query->orderBy('score', 'desc');
        }
    }

    /**
     * Set up the language
     *
     * @throws Exception
     */
    protected function setupLanguage(): void
    {
        if (is_null($this->query)) {
            throw new Exception('Init search query not set');
        }

        if (
            isset($this->filters['langcode'])
        ) {
            $this->query->where('langcode', [null, $this->filters['langcode']]);
        }
    }

    /**
     * Apply basic filters
     *
     * @throws Exception
     */
    protected function applyFilters(): void
    {
        if (is_null($this->query)) {
            throw new Exception('Init search query not set');
        }

        if (array_key_exists('global', $this->filters) && $this->filters['global']) {
            // On a global search, we'll only search in global searchable classes
            $globalSearchable = Searchable::getGlobalSearchableClasses();
            $in = [];
            foreach ($globalSearchable as $type => $class) {
                // Check if front want only certain global searchable types
                if (array_key_exists('in', $this->filters) && ! empty($this->filters['in'])) {
                    if (in_array($type, $this->filters['in'])) {
                        $in[] = $type;
                    }
                } else {
                    $in[] = $type;
                }
            }
            $this->filters['in'] = $in;
            $this->query->where('_global', true);
        }

        // In which indexes to search
        if (array_key_exists('in', $this->filters)) {
            $in = collect($this->filters['in']);

            // Special _contents filter ( means all searchable types )
            if ($in->count() == 1 && $in->first() == '_contents') {
                if (array_key_exists('global', $this->filters) && $this->filters['global']) {
                    $in = collect(Schema::getGlobalSearchableContentTypes());
                } else {
                    $in = collect(Schema::getSearchableContentTypes());
                }
            }

            if ($in->isNotEmpty()) {
                $types = inside_search_get_class_from_filters($in);

                if ($types->count() != $in->count()) {
                    // Means filter is false
                    throw new Exception('Wrong filters');
                }

                $key = 'indexable_type';

                if (count($types) > 1) {
                    $key = 'indexable_type IN';
                    $indexableType = '["'.implode('","', $types->toArray()).'"]';
                } else {
                    $indexableType = $types->first();
                }

                $this->query->where($key, $indexableType);
            }
        }

        if (array_key_exists('$or', $this->filters)) {
            $this->query->where('or', $this->filters['$or']);
        }

        $dateFields = ['created_at', 'updated_at', 'published_at', 'publishable_at'];
        foreach ($dateFields as $dateField) {
            if (! array_key_exists($dateField, $this->filters)) {
                continue;
            }

            $filters = $this->filters[$dateField];

            if (! is_array($filters) || ! isset($filters[0])) {
                $filters = [$filters];
            }

            foreach ($filters as $filter) {
                $key = $dateField;
                $operator = $filter['operator'];
                $value = $filter['value'];

                if (config('scout.driver', 'mysql') == 'solr') {
                    $key .= ':'.$operator;
                } else {
                    switch ($operator) {
                        case 'lt':
                            $key .= ' <';
                            break;
                        case 'lte':
                            $key .= ' <=';
                            break;
                        case 'gt':
                            $key .= ' >';
                            break;
                        case 'gte':
                            $key .= ' >=';
                            break;
                    }
                }

                if (Str::lower($value) == 'now()') {
                    $value = now();
                }

                $this->query->where($key, $value);
            }
        }

        $searchableContentTypes = Schema::getSearchableContentTypes();

        foreach ($this->filters as $saerchableContentType => $condition) {
            if (
                is_array($condition) &&
                ! empty($condition['uuid:eq']) &&
                ($saerchableContentType === 'authors' || in_array($saerchableContentType, $searchableContentTypes))
            ) {
                $this->postFilter[$saerchableContentType] = type_to_class($saerchableContentType)::find($condition['uuid:eq'])?->uuid_host;
            }
        }
    }

    /**
     * Apply custom filters
     *
     * @throws Exception
     */
    protected function applyCustomFilters(): void
    {
        if (is_null($this->query)) {
            throw new Exception('Init search query not set');
        }
        // Apply custom filter
        $types = [];
        $excludedFilter = array_flip([
            'limit',
            'order',
            'in',
            'langcode',
            'global',
            'publishable_at',
            'created_at',
            'updated_at',
        ]);
        $customFilters = array_diff_key($this->filters, $excludedFilter);

        if (count($customFilters) == 0) {
            return;
        }

        if (array_key_exists('in', $this->filters)) {
            $types = collect($this->filters['in'])->toArray();
        }

        if (count($types) == 0) {
            $types = SchemaHelper::listContentTables();

            foreach ($types as &$type) {
                $type = str_replace('inside_content_', '', $type);
            }
        }

        if (count($types) >= 1) {
            $type = Arr::first($types);
            try {
                // Use first type to get fields
                Schema::getModelOptions($type);
            } catch (ModelSchemaNotFoundException $e) {
                return;
            }
            $filters = collect();
            try {
                foreach ($customFilters as $key => $value) {
                    // Get field
                    $name = $key;
                    $operator = null;
                    if (strpos($key, ':')) { // If front asked with operator
                        [$name, $operator] = explode(':', $key, 2);
                    }
                    if (isset($operator) && ! in_array($operator, ['eq', 'in'])) {
                        throw new RuntimeException('operator not valid on filter ['.$name.']');
                    }
                    $fieldOptions = null;
                    try {
                        $fieldOptions = Schema::getFieldOptions($type, $name);
                    } catch (Exception $e) {
                        if (! in_array($name, ['archived', 'workflow_status', 'uuid'])) {
                            continue;
                        }

                        if ($name === 'archived' && ! Package::has('inside-archive')) {
                            continue;
                        }

                        if ($name === 'workflow_status' && ! Package::has('inside-workflow')) {
                            continue;
                        }
                    }

                    if (
                        (isset($fieldOptions['searchable_filter']) && $fieldOptions['searchable_filter'])
                        || // Hard drupal field
                        in_array($name, ['status', 'archived', 'workflow_status', 'uuid'])
                    ) {
                        // The field is filterable let's filter !
                        if (config('scout.driver', 'mysql') == 'solr') {
                            if ($name === 'uuid') {
                                $uuids = collect($value)
                                  ->flatten()
                                  ->filter()
                                  ->map(fn (string $uuid) => "\"$uuid\"")
                                  ->join(',');

                                $this->query->where(
                                    'id IN',
                                    "[$uuids]"
                                );
                                continue;
                            } elseif (in_array($name, ['status', 'archived'])) {
                                $this->query->where($name, ((bool) $value ? 'true' : 'false'));
                                continue;
                            } else {
                                if (in_array($name, ['title', 'langcode', 'url', 'uuid'])) {
                                    $this->query->where($name, $value);
                                    continue;
                                }

                                if ($name === 'workflow_status') {
                                    $this->query->where($name.'_int', (int) $value);
                                }
                            }
                            if (isset($fieldOptions) && in_array($fieldOptions['type'], ['boolean', 'checkbox'])) {
                                $this->query->where($name.'_boolean', ((bool) $value ? 'true' : 'false'));
                                continue;
                            }
                            if (isset($fieldOptions) && $fieldOptions['type'] == 'reference') {
                                $uuids = collect(['uuid', 'uuid:eq', 'uuid:in'])
                                  ->map(fn (string $key) => $value[$key] ?? null)
                                  ->flatten()
                                  ->filter()
                                  ->unique()
                                  ->values();

                                if ($uuids->isEmpty()) {
                                    continue;
                                }

                                $languages = collect(list_languages());
                                $target = $fieldOptions['target'][0];
                                $table = type_to_table($target);
                                $uuids = type_to_class($target)::query()
                                  ->whereIn("$table.uuid", $uuids) // We put tale name in order to not conflict with permission scope
                                  ->get()
                                  ->flatMap(fn (Content $content) => $languages->map(fn (string $lang) => $content->getTranslationIfExists($lang)->uuid))
                                  ->unique()
                                  ->values();

                                if ($uuids->isEmpty()) {
                                    continue;
                                }

                                $uuids = $uuids->map(fn (string $uuid) => "\"$uuid\"")->join(',');

                                $this->query->where(
                                    "{$type}_{$name}_reference IN",
                                    "[$uuids]"
                                );

                                continue;
                            }

                            $this->query->where($name.'_text', (string) $value);
                            continue;
                        }
                        // Note: where on scout queries doesn't have operator. We use special trick with
                        // name => name operator
                        // Inside Search Database will recognize this format
                        //   $this->query->where('filter like', '%' . $name . ':' . $value . '%');
                        if (is_array($value)) {
                            if (array_key_exists('uuid', $value)) {
                                $value = $value['uuid'];
                            } elseif (array_key_exists('uuid:eq', $value)) {
                                $value = $value['uuid:eq'];
                            } elseif (array_key_exists('uuid:in', $value)) {
                                $value = $value['uuid:in'];
                            }
                        }
                        $filters[$name] = $value;
                    }
                }
            } catch (ModelNotFoundException $e) {
                // Failed to load fields no filtering
            }

            $imploded = $filters->transform(
                function ($value, $key) {
                    if (is_array($value)) {
                        $filter = '';
                        // Front asked for many uuids
                        if (isset($value['uuid:eq']) && is_string($value['uuid:eq'])) {
                            $value = [$value['uuid:eq']];
                        } elseif (isset($value['uuid']) && is_string($value['uuid'])) {
                            $value = [$value['uuid']];
                        } elseif (isset($value['uuid:in']) && is_array($value['uuid:in'])) {
                            $value = $value['uuid:in'];
                        }
                        foreach ($value as $uuid) {
                            $filter .= '%'.$key.':'.$uuid.'%|';
                        }

                        return substr($filter, 0, -1);
                    }

                    return '%'.$key.':'.$value.'%';
                }
            )->implode('&');

            if (! empty($imploded)) {
                $this->query->where('filter like', $imploded);
            }
        }
    }

    /**
     * Get search result for a content type
     */
    protected function getSearchResultFields(string $type, int $depth = 0): array
    {
        if ($depth > 3) {
            return [];
        }

        $contentFields = ['content_type', 'uuid', 'admin'];

        $options = Schema::getModelOptions($type);
        if (array_key_exists('aliasable', $options) && $options['aliasable']) {
            $contentFields[] = 'slug';
        }

        if ($type != 'users' && $depth === 0) {
            // Check if users has a field referencing $type ?
            $userFields = Schema::getFieldListing('users');
            foreach ($userFields as $userField) {
                if ($userField != 'author') {
                    $options = Schema::getFieldOptions('users', $userField);
                    if ($options['type'] == 'reference' && in_array($type, $options['target'])) {
                        // Found a reference to $type
                        $contentFields[] = [
                            'users' => [
                                'filters' => ['reverse' => true],
                                'fields' => $this->getSearchResultFields(
                                    'users',
                                    100 // We don't wan't to go more in depth
                                ),
                            ],
                        ];
                    }
                }
            }
        }

        foreach (Schema::getFieldListing($type) as $field) {
            $definition = Schema::getFieldOptions($type, $field);
            if (array_key_exists('search_result_field', $definition) && $definition['search_result_field']) {
                if ($definition['type'] == 'reference') {
                    if (
                        (! in_array($definition['target'][0], ['users', 'author']))
                        || (! in_array($field, ['authors', 'author']))
                    ) {
                        $contentFields[] = [
                            $field => $this->getSearchResultFields($definition['target'][0], ($depth + 1)),
                        ];
                    } else {
                        $contentFields[] = [
                            'authors' => array_intersect(
                                array_filter(
                                    $this->getSearchResultFields($definition['target'][0], ($depth + 1)),
                                    function ($value) {
                                        return is_string($value);
                                    }
                                ),
                                ['uuid', 'firstname', 'lastname', 'email', 'image']
                            ),
                        ];
                    }
                } else {
                    $contentFields[] = $field;
                }
            }
        }

        if (Schema::hasFieldOfType($type, 'section')) {
            $contentFields[] = 'has_section';
        }

        return $contentFields;
    }

    /**
     * Transform our items with only wanted fields
     */
    protected function applyCustomTransformers(Collection $items): Collection
    {
        $items = $this->applyPostFilter($items);

        $transformer = new ContentTransformer();
        $transformer->setFilters($this->filters);

        return $items->transform(
            function ($item) use ($transformer) {
                $fields = $this->fields;

                if (empty($fields)) {
                    $type = class_to_type($item);
                    // Get search result field for this type
                    if (! array_key_exists($type, $this->contentFields)) {
                        $this->contentFields[$type] = $this->getSearchResultFields($type);
                        $this->contentFields[$type][] = 'score';
                        $this->contentFields[$type][] = 'url';
                        $this->contentFields[$type][] = 'slug';
                        $this->contentFields[$type][] = 'published_at';
                        $this->contentFields[$type][] = 'updated_at';
                    }

                    $fields = $this->contentFields[$type];
                }

                return $transformer->transform($item, $fields);
            }
        );
    }

    public function specialSearch(): array
    {
        if (config('scout.driver', 'mysql') != 'solr') {
            abort(400, 'driver not support');
        }

        /** @var User $user */
        $user = Auth::user();

        //Content type for the search
        $contentTypes = $this->request->get('content_types');
        if ($contentTypes) {
            $contentTypes = explode(',', $contentTypes);
        } else {
            // Add all searchable content type
            $contentTypes = Schema::getContentTypes(
                function ($model) {
                    return array_key_exists('searchable', $model['options']) && $model['options']['searchable']
                    && array_key_exists('global_searchable', $model['options'])
                    && $model['options']['global_searchable'];
                }
            );
        }

        // Searched terms ( from input text )
        $terms = $this->request->get('terms', '');
        /** NOTE:
         * Front like to search for ... nothing ( empty string, that empty string middleware turn to null ):
         * goal was to get everything
         */
        if (is_null($terms)) {
            $terms = '';
        }

        // Pagination
        $perPage = $this->request->get('perPage', 10);
        $page = $this->request->get('page', 1);

        // Sort
        $sorts = $this->request->get('sort', 'score:desc');
        $sorts = explode(',', $sorts);

        // withTrashed ?
        $withTrash = $user->isSuperAdmin() && $this->request->get('withTrash', false);

        // We don't use Scout here but our special service
        $filters = $this->request->get('filters', '[]');
        if (($filters = json_decode(stripslashes(trim($filters)), true)) === null) {
            $filters = [];
        }

        $hasCustomScope = ($callback = config('permission.global_scope')) && is_callable($callback);

        $toZap = 0;
        if ($hasCustomScope) {
            $searchKey = md5(
                implode('-', $contentTypes).'-'.$terms.'-'.
                implode('-', $filters).'-'.implode('_', $sorts).'-'.$perPage
                .'-'.$withTrash
            );
            $toZap = 0;
            if (Cookie::has('_search_cursor')) {
                // Decode last pagination
                /** @var array $pagination */
                $pagination = app('encrypter')->decrypt(Cookie::get('_search_cursor'));
                if (isset($pagination[$searchKey])) {
                    $currentSearch = $pagination[$searchKey];
                    // Get information for current search parameters
                    if (isset($currentSearch[$page])) {
                        // Change cursor according to last pagination
                        $page = $currentSearch[$page]->page ?? 1;
                        $toZap = $currentSearch[$page]->cursor ?? 0; // Last page we already got x element of $page
                    }
                }
            }
        }

        $count = $cursor = 0;

        $finalResult = null;
        $usingPage = $page;
        do {
            $result = app(EngineManager::class)->engine()->specialSearch(
                $contentTypes,
                $terms,
                $filters,
                $sorts,
                $perPage,
                $usingPage,
                $withTrash
            );

            if ($hasCustomScope) {
                $initialCount = count($result['data']); // Use to stop loop if we finished search
                if (is_array($result['data'])) {
                    $cursor = 0;
                    foreach ($result['data'] as $uuid => $data) {
                        if ($count >= $perPage) {
                            unset($result['data'][$uuid]);
                            continue; // We already got every data we wanted
                        }
                        if ($toZap-- > 0) {
                            $cursor++;
                            unset($result['data'][$uuid]);
                            continue; // zapped already returned result
                        }

                        // Try to get result to check we are allowed to view that result
                        $item = call_user_func(type_to_class($data['type']).'::find', $uuid);
                        if ($item) {
                            $cursor++;
                            $count++;
                            continue; // Yeah, I can see that !
                        }

                        // Problem there !
                        unset($result['data'][$uuid]);
                    }
                    if ($count < $perPage) {
                        $usingPage++;
                    }
                }
            }
            if ($finalResult === null) {
                $finalResult = $result;
            } else {
                $finalResult['data'] = array_merge($finalResult['data'], $result['data']);
            }
        } while ($hasCustomScope && ($count < $perPage) && ($initialCount > 0));

        if ($hasCustomScope && isset($searchKey)) {
            if (! isset($pagination)) {
                $pagination = [];
            }
            // Save cookie for next page !
            if (! isset($pagination[$searchKey]) || ! is_array($pagination[$searchKey])) {
                $pagination[$searchKey] = [];
            }

            if ($cursor == $count) { // we got all needed item, we should move to next page then!
                $usingPage++;
                $cursor = 0;
            }

            $pagination[$searchKey][$page + 1] = (object) [
                'page' => $usingPage,
                'cursor' => $cursor,
            ];
            // Decode last pagination
            Cookie::queue('_search_cursor', app('encrypter')->encrypt($pagination));
        }

        return $finalResult;
    }

    /**
     * Advanced search with facets !
     * @throws Exception
     */
    public function advancedSearch(): array
    {
        // Actually only solr driver support advanced search
        if (config('scout.driver', 'mysql') != 'solr') {
            abort(400, 'driver not support');
        }

        /** @var User $user */
        $user = Auth::user();

        //Content type for the search
        $contentTypes = explode(',', $this->request->get('content_types', ''));

        // Searched terms ( from input text )
        $terms = $this->request->get('terms', '');
        /** NOTE:
         * Front like to search for ... nothing ( empty string, that empty string middleware turn to null ):
         * goal was to get everything
         */
        if (is_null($terms)) {
            $terms = '';
        }

        // Pagination
        $perPage = $this->request->get('perPage', 10);
        $page = $this->request->get('page', 1);

        // Sort
        $sorts = $this->request->get('sort', 'score:desc');
        $sorts = explode(',', $sorts);

        // withTrashed ?
        $withTrash = $user->isSuperAdmin() && $this->request->get('withTrash', false);

        $language = Users::findOrFail($user->uuid)->langcode;

        // We don't use Scout here but our special service
        $filters = $this->request->get('filters', '[]');
        if (($filters = json_decode(stripslashes(trim($filters)), true)) === null) {
            $filters = [];
        }

        $engineResults = app(EngineManager::class)->engine()->advancedSearch(
            $contentTypes,
            $terms,
            $filters,
            $sorts,
            $perPage,
            $page,
            $withTrash
        );

        // Let's get contents
        $transformer = new ContentTransformer();

        // Prepare results
        $results = [
            'total' => $engineResults['total'],
            'perPage' => $perPage,
            'page' => $page,
            'maxPage' => max((int) ceil($engineResults['total'] / $perPage), 1),
            'maxScore' => $engineResults['maxScore'],
            'items' => [],
            'filters' => [],
        ];

        /** @var Model $model */
        $model = Model::where('class', type_to_class($contentTypes[0]))->with('fields')->first();
        $fields = $model->fields->pluck('options', 'name');

        // Filters
        $engineFilters = [];
        foreach ($engineResults['filters'] as $filter => &$counts) {
            $filterResults = [];
            $target = Arr::first($fields[$filter]['target']);
            if ($target === null) {
                continue;
            }
            $hasWeight = Schema::hasField($target, 'weight');
            foreach ($counts as $uuid => $count) {
                $item = call_user_func(type_to_class($target).'::find', $uuid);
                if ($item) {
                    $filterResult = [
                        'uuid' => $uuid,
                        'count' => $count,
                        'title' => $item->title,
                        'slug' => $item->slug[0] ?? null,
                    ];
                    if ($hasWeight) {
                        $filterResult['weight'] = $item->weight;
                    }
                    if (Schema::hasField($target, 'image')) {
                        $filterResult['image'] =
              File::exists(Storage::path('app/'.$item->image)) ? Storage::url('app/'.$item->image)
                : null;
                    }
                    if (Schema::hasField($target, 'color')) {
                        $filterResult['color'] = $item->color;
                    }
                    $filterResults[] = $filterResult;
                }
            }
            if (count($filterResults) > 0) {
                $options = Schema::getFieldOptions($contentTypes[0], $filter);
                $engineFilters[$filter] = [
                    'values' => $this->getFilterValue($filters, $filter),
                    'title' => $options['title'][$language] ?? $filter,
                    'widget' => $options['filter_widget'],
                    'category' => $options['filter_category'] ?? '',
                    'order' => $options['filter_order'] ?? 1,
                    // Order filter results by title instead of score ( default solr one )
                    'options' => array_values(
                        Arr::sort(
                            $filterResults,
                            function ($filter) use ($hasWeight) {
                                return $hasWeight ? $filter['weight'] : $filter['title'];
                            }
                        )
                    ),
                ];
            }
        }

        // Rearrange engineFilters to fit front needs
        $currentCategory = '';
        foreach (Arr::sort($engineFilters, 'order') as $filter => $values) {
            if (! empty($values['category'])) {
                if ($values['category'] != $currentCategory) {
                    // it's a categorised filter and a new one !
                    $currentCategory = $values['category'];
                    $results['filters'][Str::slug($values['category'])] = [
                        'type' => 'category',
                        'title' => $values['category'],
                        'fid' => Uuid::uuid4()->toString(),
                        'children' => [
                            $filter => [
                                'type' => 'filter',
                                'title' => $values['title'],
                                'fid' => Uuid::uuid4()->toString(),
                                'values' => $values['values'],
                                'widget' => $values['widget'],
                                'options' => $values['options'],
                            ],
                        ],
                    ];
                } else {
                    // Same category, add the filter to the last added node
                    $results['filters'][Str::slug($values['category'])]['children'][$filter] = [
                        'type' => 'filter',
                        'fid' => Uuid::uuid4()->toString(),
                        'title' => $values['title'],
                        'values' => $values['values'],
                        'widget' => $values['widget'],
                        'options' => $values['options'],
                    ];
                }
            } else {
                $currentCategory = '';
                // Orphan filter
                $results['filters'][$filter] = [
                    'type' => 'filter',
                    'title' => $values['title'],
                    'fid' => Uuid::uuid4()->toString(),
                    'values' => $values['values'],
                    'widget' => $values['widget'],
                    'options' => $values['options'],
                ];
            }
        }

        // Load search results fields
        $fields = [];
        foreach ($contentTypes as $type) {
            // Store search result field for this type
            $fields[$type] = $this->getSearchResultFields($type);
        }

        // Get items from database
        foreach ($engineResults['data'] as $field => $items) {
            if (array_key_exists($field, $fields)) {
                $target = $fields[$field];
                $contents = call_user_func(type_to_class($type).'::find', array_keys($items))->keyBy('uuid');
                foreach ($items as $uuid => $score) {
                    if (isset($contents[$uuid])) {
                        $results['items'][] =
              array_merge(['score' => $score], $transformer->transform($contents[$uuid], $fields[$type]));
                    }
                }
            }
        }

        return $results;
    }

    /**
     * Get values for field filter
     *
     * Note each filter should be unique
     *
     * @param array $filters
     * @param string $field
     *
     * @return array|null
     */
    protected function getFilterValue(array $filters, string $field): ?array
    {
        foreach ($filters as $filter) {
            if (($filter['filter'] == $field) && ($filter['operator'] == 'in')) {
                return $filter['values'];
            }
            // TODO: add range operator
        }

        return null;
    }

    private function applyPostFilter(Collection $items): Collection
    {
        if (empty($this->postFilter)) {
            return $items;
        }

        return $items->filter(fn ($item) => collect($this->postFilter)->every(function ($uuid, $relation) use ($item) {
            $relationMethod = Str::camel($relation);
            if (method_exists($item, $relationMethod) && $item->{$relationMethod}() instanceof Relation) {
                return $item->{$relationMethod}()->get()->pluck('uuid_host')->contains($uuid);
            }

            $relationValue = $item->getAttribute($relation);

            return $relationValue instanceof EloquentDatabaseCollection
                ? $relationValue->pluck('uuid')->contains($uuid)
                : $relationValue?->uuid === $uuid;
        }))
            ->values();
    }

    private function rebuildPaginator(LengthAwarePaginator $result): LengthAwarePaginator
    {
        $filteredItems = $result->setCollection(
            $this->applyCustomTransformers(
                collect($result->items())
            )
        );

        return empty($this->postFilter) ? $filteredItems : new CustomLengthAwarePaginator(
            $filteredItems->forPage($result->currentPage(), $result->perPage()),
            $filteredItems->count(),
            $result->perPage(),
            $result->currentPage(),
            [
                'path' => $result->path(),
                'query' => $this->request->except('page'),
                'pageName' => $result->getPageName(),
            ]
        );
    }
}
