<?php

namespace Inside\Search\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Inside\Search\Contracts\Search as SearchContract;
use OpenApi\Annotations as OA;

class AdvancedSearch extends Controller
{
    /**
     * @throws Exception
     */
    public function __invoke(Request $request, SearchContract $service): JsonResponse
    {
        return response()->json($service->withRequest($request)->advancedSearch());
    }
}

/**
 * @OA\Schema(
 *    schema="AdvancedSearchParameterFilter",
 *        type="object",
 *        @OA\Property(
 *            property="filter",
 *            type="string",
 *            description="Le nom du filtre ( doit être un filtre avec un filter_widget )",
 *            example="materials",
 *        ),
 *        @OA\Property(
 *            property="operator",
 *            type="string",
 *            description="L'opérateur de filtre. Les choix possibles sont limités à **in** ou **range**",
 *            default="in",
 *            enum={"in","range"},
 *            example="in",
 *        ),
 *        @OA\Property(
 *            property="values",
 *            type="array",
 *            description="Les valeurs pour le filtre. Si l'opérateur est **in**, les valeurs souhaités
pour le filtre donnée. Si l'opérateur est **range** alors la valeur de début et de fin",
 *            @OA\Items(
 *               oneOf={@OA\Schema(type="string", example="51d56eb8-72e8-40ba-be15-42f6a0fd0568"),
 *               @OA\Schema(type="integer", example=1983)}
 *            )
 *        )
 * )
 *
 *
 *
 *
 * @OA\Schema(
 *     schema="AdvancedSearchResultFilterOptions",
 *     type="object",
 *     description="Une valeur de filtre",
 *     @OA\Property(
 *        property="uuid",
 *        description="L'uuid du filtre.",
 *        type="string",
 *        example="eb54d007-76b0-4aab-8bd0-40e20758c48f"
 *     ),
 *     @OA\Property(
 *        property="count",
 *        description="Le nombre de résultat si l'utilisateur choisi cette valeur.",
 *        type="integer",
 *        example=12
 *     ),
 *     @OA\Property(
 *        property="title",
 *        description="Le titre.",
 *        type="string",
 *        example="Automotive Equipment & Services"
 *     ),
 *     @OA\Property(
 *        property="slug",
 *        description="Le slug.",
 *        type="string",
 *        example="automotive-equipment-services"
 *     ),
 *     example={"uuid":"e4096dda-a730-4c55-a8e8-d86254d4cd2a","count":1,"title":"Autofast","slug":"autofast-1"}
 * )
 *
 *
 * @OA\Schema(
 *      schema="AdvancedSearchResultFilterFilter",
 *      type="object",
 *      description="Un filtre",
 *       @OA\Property(
 *           property="type",
 *           description="Le type de neud de rétultat : dans ce cas filter",
 *           type="string",
 *           enum={"filter"},
 *           example="filter"
 *       ),
 *       @OA\Property(
 *           property="title",
 *           description="Le titre du filtre",
 *           type="string",
 *           example="Mon filtre"
 *       ),
 *       @OA\Property(
 *           property="fid",
 *           description="Un uuid utilisé pour la construction du front",
 *           type="string",
 *           example="57b02be7-79dc-41d2-9ccf-092fd56cb9fc"
 *       ),
 *       @OA\Property(
 *           property="values",
 *           description="Les valeurs sélectionnés par l'utilisateur. Une valeur **null** indique que
l'utilisateur n'a rien sélectionné",
 *          type="array",
 *           nullable=true,
 *           @OA\Items(type="string",description="La liste des uuids sélectionné pour ce filtre"),
 *            example={"eb54d007-76b0-4aab-8bd0-40e20758c48f"}
 *       ),
 *      @OA\Property(
 *          property="widget",
 *           description="Le nom du widget front a utilisé pour afficher le filtre.",
 *          type="string",
 *          example="checkboxesgroup"
 *       ),
 *       @OA\Property(
 *          property="options",
 *          description="Les valeurs possibles du filtre.",
 *          type="array",
 *           @OA\Items(
 *               ref="#/components/schemas/AdvancedSearchResultFilterOptions"
 *             )
 *       )
 *  )
 *
 * @OA\Schema(
 *      schema="AdvancedSearchResultFilterCategory",
 *      type="object",
 *      description="Catégorie de filtres. Un seul niveau, les enfants ne peuvent avoir d'enfant!",
 *       @OA\Property(
 *           property="type",
 *           description="Le type de neud de rétultat dans ce cas category",
 *           type="string",
 *           enum={"category"},
 *           example="category"
 *       ),
 *       @OA\Property(
 *           property="title",
 *           description="Le titre de la catégorie",
 *           type="string",
 *           example="Ma catégorie"
 *       ),
 *       @OA\Property(
 *           property="fid",
 *           description="Un uuid utilisé pour la construction du front",
 *           type="string",
 *           example="57b02be7-79dc-41d2-9ccf-092fd56cb9fc"
 *       ),
 *       @OA\Property(
 *           property="children",
 *           description="Les enfants de cette catégorie. La clé est le nom machine du filtre.",
 *           type="object",
 *           @OA\AdditionalProperties(ref="#/components/schemas/AdvancedSearchResultFilterFilter")
 *       )
 * )
 *
 * @OA\Schema(
 *      schema="AdvancedSearchResultFilter",
 *      description="Une catégorie de filtres ou un filtre **orphelin**. La clé est soit un nom machine de
 *      catégorie ou de filtre", type="object",
 *     @OA\AdditionalProperties(
 *      oneOf={
@OA\Schema(
schema="AdvancedSearchResultFilterFilter",
ref="#/components/schemas/AdvancedSearchResultFilterFilter"
),
@OA\Schema(
schema="AdvancedSearchResultFilterCategory",
ref="#/components/schemas/AdvancedSearchResultFilterCategory"
)
}
)
 *  )
 *
 * @OA\Schema(
 *     schema="AdvancedSearchResult",
 *     type="object",
@OA\Property(
property="total",
description="Le nombre total de résultat.",
type="integer",
minimum=0,
example=12
),
@OA\Property(
property="perPage",
description="Le nombre de résultat par page.",
type="integer",
minimum=1,
example=10
),
@OA\Property(
property="page",
description="Le uméro de la page en cours.",
type="integer",
minimum=1,
example=2
),
@OA\Property(
property="maxPage",
description="Le numéro de la dernière page.",
type="integer",
minimum=1,
example=21
),
@OA\Property(
property="maxScore",
description="Le score max obtenu dans les résultats.",
type="number",
format="float",
minimum=0.1,
example=7.745883
),
@OA\Property(
property="items",
description="Les résultats de la recherche.",
type="array",
@OA\Items(
type="object",
@OA\Property(
property="score",
description="Le score obtenu pour ce résultat.",
type="number",
format="float",
minimum=0.1,
example=7.4085317
),
@OA\Property(
property="content_type",
description="Le type de contenu du résultat.",
type="string",
example="news"
),
@OA\Property(
property="uuid",
description="L'uuid du résultat.",
type="string",
example="28869772-6ac6-4eda-a9b7-8e3a55d46749"
),
additionalProperties=true
)
),
@OA\Property(
property="filters",
description="Les filtres de recherche.",
ref="#/components/schemas/AdvancedSearchResultFilter"
)
 *  )
 *
 * @OA\Get(
 *      path="/advanced/search",
 *      operationId="advancedSearch",
 *      tags={"Recherche"},
 *      summary="Effectue une recherche avancée en utilisant des filtres ( sur le modèle d'un moteur à facettes
     *      ).",
     * @OA\Parameter(
 *         name="content_types",
 *         in="query",
 *         explode=false,
 *        description="Les types de contenus dans lesquels s'effectue la recherche avancée.",
 *        @OA\Schema(
 *            type="array",
 *             @OA\Items(
 *                type="string",
 *                description="Type (nom inside) de contenu dans lequel s'effectue la recherche avancée."
 *            ),
 *            example={"news","events"}
 *        )
 *       ),
 * @OA\Parameter(
 *           name="terms",
 *          in="query",
 *           description="Le ou les termes recherchés",
 *           @OA\Schema(
 *               type="string",
 *              example="Ma recherche"
 *          )
 *       ),
 * @OA\Parameter(
 *           name="perPage",
 *           in="query",
 *           description="Le nombre de résultat à retourner",
 *           @OA\Schema(
 *               type="integer",
 *               minimum=1,
 *                default=10,
 *                example=20
 *            )
 *       ),
 * @OA\Parameter(
 *           name="page",
 *           in="query",
 *          description="La page désirée",
 *          @OA\Schema(
 *             type="integer",
 *              minimum=1,
 *              default=1,
 *             example=2
 *         )
 *      ),
 * @OA\Parameter(
 *          name="sort",
 *           in="query",
 *           description="Le tri des résultats désiré. Une de colonne incorrecte empechera d'obtenir des
résultats. Le sens par défaut du tri est ascendant. Pour indiquer un sens, il suffit d'ajouter après le nom
 *   du champ ( :asc ou :desc ). Pour indiquer plusieurs tris, il suffit de séparer par une virgule.   ",
 * @OA\Schema(
 *              type="string",
 *               default="score:desc",
 *                example="title"
 *           )
 *       ),
 * @OA\Parameter(
 *             name="withTrash",
 *            in="query",
 *          description="Indique si l'on souhaite les contenus effacés également ( le paramètre n'est traité que
 * si on est super admin )",
 * @OA\Schema(
 *              type="boolean",
 *              default=false,
 *              example=false
 *          )
 *       ),
 * @OA\Parameter(
 *           name="filters",
 *           in="query",
 *           description="Les filtres sélectionnés. Le filtre est au format json. Le schema s'appel
AdvancedSearchParameterFilter voir la documentation.
`[{ ""filter"": ""materials"", ""operator"": ""in"", ""values"":[ ""51d56eb8-72e8-40ba-be15-42f6a0fd0568"" ] }]`
",
 *           required=false,
 *           @OA\Schema(
 *             type="array",
 *             format="json",
 *             @OA\Items(ref="#/components/schemas/AdvancedSearchParameterFilter"),
 *             example={{"filter":"materials","operator":"in","values":{"51d56eb8-72e8-40ba-be15-42f6a0fd0568"}}}
 *           ),
 *       ),
 *
 * @OA\Response(
 *          response=200,
 *          description="Les informations de la recherche et ses résultats",
 *          @OA\JsonContent(ref="#/components/schemas/AdvancedSearchResult")
 *       ),
 * @OA\Response(response=400, description="Bad request")
 *     )
 */
