<?php

namespace Inside\Search\Console;

use Illuminate\Console\Command;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Str;
use Inside;
use Inside\Facades\Package;
use Inside\Search\Facades\Searchable;

/**
 * Class IndexMissingContentCommand
 *
 * @category Class
 * @author   Maecia <contact@feldoe.net>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class IndexMissingContentCommand extends Command
{
    /**
     * @var string
     */
    protected $name = 'index:rebuild:missing';

    /**
     * @var string
     */
    protected $signature = 'index:rebuild:missing';

    /**
     * @var string
     */
    protected $description = 'Reindex missing contents';

    /**
     * Handle index command
     *
     * @return void
     * @throws \Exception
     */
    public function handle(): void
    {
        $now = now();
        Log::info('Starting to index at ['.$now->toFormattedDateString().' '.$now->toTimeString().']');

        $searchables = Searchable::getSearchableClasses();
        $searchables->pluck('class')->each(function ($modelClass) {
            $type = class_to_type($modelClass);
            Log::info("Indexing [$type]");
            $uuids = $this->getIndexedUuids($modelClass);
            $contentsNotIndexed = $modelClass::query()
              ->where('status', true)
              ->whereNotIn('uuid', $uuids)
              ->get();

            $contentsNotIndexed->searchable();
            Log::info(sprintf('[%s] %d elements indexed!', $type, $contentsNotIndexed->count()));
        });
    }

    private function getIndexedUuids(string $class): Collection
    {
        return (config('scout.driver', 'mysql') !== 'solr') ? $this->getDatabaseIndexedUuids($class) : $this->getSolrIndexedUuids($class);
    }

    private function getSolrIndexedUuids(string $class): Collection
    {
        $query = call_user_func(search_namespace('Models\Index::search'));
        $query->where('indexable_type', $class);

        // can't retrieving all records from solr we should use paginate with big value
        return $query->within('*')->paginate(900000000)->pluck('uuid');
    }

    private function getDatabaseIndexedUuids(string $class): Collection
    {
        $query = call_user_func(search_namespace('Models\Index::query'));

        return $query->where('indexable_type', $class)->get()->pluck('indexable_id');
    }
}
