<?php

namespace Inside\Reaction\Services;

use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Collection as SupportCollection;
use Illuminate\Support\Facades\Log;
use Inside\Content\Contracts\Transformer;
use Inside\Content\Facades\ContentHelper;
use Inside\Content\Models\Contents\Users;
use Inside\Database\Eloquent\Builder;
use Inside\Reaction\Exceptions\ReactionNotFoundException;
use Inside\Reaction\Exceptions\ReactionValidatorException;
use Inside\Reaction\Exceptions\UniqueReactionException;
use Inside\Reaction\Models\Reaction;
use Inside\Reaction\Validators\ReactionValidator;

class ReactionService
{
    protected array $formatted = [];

    public function __construct(
        protected Transformer $transformer,
        protected ReactionValidator $validator
    ) {
    }

    /**
     * @throws ReactionValidatorException
     * @throws UniqueReactionException
     */
    public function validate(array $data): void
    {
        $this->validator->validate($data);
    }

    public function format(Reaction $reaction, array $fields = []): void
    {
        $this->formatted = ['data' => $this->transformer->transform($reaction, $fields)];
    }

    public function formatCollection(Collection|SupportCollection|Paginator $reactions, array $fields = []): void
    {
        if ($reactions instanceof Paginator) {
            $reactions = $reactions->getCollection();
        }

        $this->formatted = $this->transformer->transformCollection($reactions, $fields);
        $this->formatted['data'] = array_values(
            array_filter(
                $this->formatted['data'],
                function ($data) {
                    return ! empty($data);
                }
            )
        );
    }

    public function getFormatted(): array
    {
        return $this->formatted;
    }

    public function list(
        string $type,
        string $related,
        array $filters = [],
        array $fields = []
    ): array {
        /** @var Builder $query */
        $query = Reaction::where('reactionable_uuid', $related)->where('type', $type);

        ContentHelper::applySortToQuery($query, $filters, 'created_at', 'desc');
        /**
         * @var LengthAwarePaginator $reactions
         */
        $reactions = ContentHelper::getResultFromQuery($query, $filters, true);

        $this->formatCollection($reactions, $fields);

        return $this->paginateCollection($reactions);
    }

    public function userList(
        string $type,
        string $related,
        int $limit,
        string $langcode = null,
        array $fields = [],
        array $reactionableTypes = [],
        int $status = 1
    ): array {
        $reactionableTypes = collect($reactionableTypes)->map(fn ($reactionableType) => type_to_class($reactionableType));
        $hasUsersReactionableType = $reactionableTypes->contains(Users::class);

        $query = Reaction::where('user_uuid', $related)->where('type', $type);
        if (! $reactionableTypes->isEmpty()) {
            $query->whereIn('reactionable_type', $reactionableTypes->all());
        }

        $query->where(function ($query) use ($langcode, $hasUsersReactionableType, $reactionableTypes) {
            $query->where(function ($query) use ($langcode) {
                if ($langcode) {
                    $query->where('langcode', $langcode);
                }
            });

            if ($hasUsersReactionableType || $reactionableTypes->isEmpty()) {
                $query->orWhere('reactionable_type', Users::class);
            }
        });

        $query->whereHasMorph('reactionable', '*', function ($query) {
            $query->where('status', 1);
        });

        $reactionsQuery = $query->with('reactionable');
        if ($limit) {
            $reactionsQuery = $reactionsQuery->limit($limit);
        }

        $reactions = $reactionsQuery->get();
        $contents = new Collection($reactions->sortByDesc('reactionable.created_at')->pluck('reactionable'));
        $this->formatCollection($contents, $fields);

        return $this->getFormatted();
    }

    public function get(
        string $type,
        string $reactionableUuid,
        string $user,
        ?string $langcode = null,
        ?string $reactionableType = null,
        array $fields = [],
        bool $formated = true
    ): array|Collection {
        $query = Reaction::where('reactionable_uuid', $reactionableUuid)
            ->where('user_uuid', $user)
            ->where('type', $type);

        if ($langcode) {
            $query->where(
                function ($query) use ($langcode) {
                    $query->where('langcode', $langcode)->orWhere('reactionable_type', Users::class);
                }
            );
        }

        if ($reactionableType) {
            $query->where('reactionable_type', $reactionableType);
        }

        $reactions = $query->get();
        $this->formatCollection($reactions, $fields);

        return $formated ? $this->getFormatted() : $reactions;
    }

    /**
     * @throws UniqueReactionException
     * @throws ReactionValidatorException
     */
    public function create(array $data, array $fields = [], bool $formatted = true): array
    {
        $this->validate($data);
        $content = call_user_func($data['reactionable_type'].'::findOrFail', $data['reactionable_uuid']);
        $data['langcode'] = $content->langcode;
        $reaction = Reaction::create($data);

        Log::debug(
            'Reaction successfully created',
            [
                'type' => $reaction->type,
                'uuid' => $reaction->user_uuid,
                'related_uuid' => $reaction->reactionable_uuid,
                'related_type' => $reaction->reactionable_type,
            ]
        );

        $this->format($reaction, $fields);

        return $formatted ? $this->getFormatted() : $reaction->toArray();
    }

    /**
     * @throws ReactionNotFoundException
     */
    public function delete(string $type, string $related, string $user, string $langcode = null): void
    {
        /** @var Collection $reactions */
        $reactions = $this->get($type, $related, $user, $langcode, null, ['uuid'], false);

        if ($reactions->isEmpty()) {
            throw new ReactionNotFoundException('Reaction not found', 400);
        }

        foreach ($reactions as $reaction) {
            $reaction->delete();
        }

        Log::debug(
            'Reaction successfully deleted',
            [
                'type'    => $type,
                'uuid'    => $user,
                'related' => $related,
            ]
        );
    }

    private function paginateCollection(LengthAwarePaginator $reactions): array
    {
        return [
            'data' => $this->getFormatted()['data'],
            'current_page' => $reactions->currentPage(),
            'last_page' => $reactions->lastPage(),
            'per_page' => $reactions->perPage(),
            'total' => $reactions->total(),
        ];
    }
}
