<?php

namespace Inside\Reaction\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Contents\Users;
use Inside\Database\Eloquent\WithEnhancedBuilder;
use Inside\Reaction\Events\ReactionSubscribedEvent;
use Inside\Reaction\Events\ReactionUnsubscribedEvent;

/**
 * @property string $reactionable_type
 * @property-read User $user
 * @property int $created_at
 * @property int $updated_at
 * @property string|null $type
 * @property object $reactionable
 * @property int $id
 * @property string $user_uuid
 * @property string $langcode
 * @property-read string $reactionable_uuid
 *
 * @method static Builder whereHas($relation, \Closure $callback = null, $operator = '>=', $count = 1)
 * @method static \Inside\Database\Eloquent\Builder where($column, $operator = null, $value = null, $boolean = 'and')
 * @method static static firstOrCreate(array $attributes, array $values = [])
 * @method static \Inside\Database\Eloquent\Builder|Reaction whereType($value)
 *
 * @mixin Builder
 * @mixin \Eloquent
 */
class Reaction extends Model
{
    use WithEnhancedBuilder;

    /**
     * @var string
     */
    protected $table = 'inside_reactions';

    /**
     * @var array
     */
    protected $fillable = [
        'user_uuid',
        'type',
        'langcode',
        'reactionable_uuid',
        'reactionable_type',
    ];

    /**
     * @var array
     */
    protected $hidden = [
        'id',
        'updated_at',
    ];

    /**
     * @var array
     */
    protected $guarded = [
        'id',
        'updated_at',
        'created_at',
    ];

    /**
     * @var array
     */
    protected $dates = ['created_at', 'updated_at'];

    /**
     * @var array
     */
    protected $casts = [
        'created_at' => 'timestamp',
        'updated_at' => 'timestamp',
    ];

    public static function boot(): void
    {
        parent::boot();

        self::created(
            function (self $model) {
                ReactionSubscribedEvent::dispatch($model);
            }
        );
        static::deleting(
            function (self $model) {
                ReactionUnsubscribedEvent::dispatch($model);
            }
        );
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(Users::class, 'user_uuid', 'uuid'); // @phpstan-ignore-line
    }

    /** @deprecated Should not be plural ... */
    public function users(): BelongsTo
    {
        return $this->user();
    }

    public function authenticationUsers(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_uuid', 'uuid');
    }

    public function reactionable(): MorphTo
    {
        return $this->morphTo('reactionable', 'reactionable_type', 'reactionable_uuid');
    }
}
