<?php

namespace Inside\Permission\Services;

use Exception;
use Illuminate\Contracts\Pagination\Paginator;
use Illuminate\Database\QueryException;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Lang;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\Schema;
use Inside\Content\Models\Content;
use Inside\Content\Models\Contents\ImageStyles;
use Inside\Content\Models\Contents\Users;
use Inside\Permission\Events\RoleAttachedEvent;
use Inside\Permission\Events\RoleDetachedEvent;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Models\Role;
use Inside\Permission\Models\RoleCategory;
use Inside\Permission\Scopes\AllowedScope;

/**
 * Service handle any role
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class RoleService
{
    /**
     * Check if we can do this action on this model
     * NOTE: this helper does not get override
     *
     * @param string $action
     * @param Content|null $model
     * @param int|null $roleId
     * @return bool
     */
    public function can(string $action = 'read', Content $model = null, int $roleId = null): bool
    {
        if (! $model || is_null($roleId)) {
            return false;
        }

        if (Permission::isSystemV2Enabled()) {
            /** @var Role $role */
            $role = Role::find($roleId);

            return $role->getAccessRestriction()->canReadContent($model);
        }

        return DB::table('inside_permissions')->where('inside_permissions.action', $action)->where(
            'inside_permissions.type',
            get_class($model)
        )->when($model->uuid, function ($query) use ($model) {
            $query->where('inside_permissions.uuid', $model->uuid);
        }, function ($query) {
            $query->whereNull('inside_permissions.uuid');
        })->where('inside_permissions.role_id', $roleId)->exists();
    }

    /**
     * Generate role category / role  tree
     *
     * @param mixed $root
     * @return array
     */
    public function tree($root = null): array
    {
        $tree = [];

        $this->node($tree, $root);

        return $tree;
    }

    /**
     * Get the node
     *
     * @param array $tree
     * @param mixed $root
     * @return void
     */
    public function node(array &$tree = [], $root = null)
    {
        $filter = ['pid' => null];
        $nodes = new Collection();

        if (! $root) {
            $nodes = $nodes->merge(RoleCategory::where($filter)->get());
            $nodes = $nodes->merge(
                Role::where(array_merge($filter, ['role_category_id' => null]))->whereNull('type')->get()
            );
        }

        if ($root) {
            $filter = ['pid' => $root->id];
        }

        if ($root && get_class($root) == RoleCategory::class) {
            $nodes = $nodes->merge(RoleCategory::where($filter)->get());
            $nodes = $nodes->merge(Role::where(['role_category_id' => $root->id])->whereNull('type')->get());
        }

        if ($root && get_class($root) == Role::class) {
            $nodes = Role::where($filter)->whereNull('type')->get();
        }

        $nodes = $nodes->sortBy('weight');

        foreach ($nodes as $node) {
            $array = $node->toArray();
            $type = 'role';

            if (get_class($node) == RoleCategory::class) {
                $type = 'role_category';
                $array['label'] = \Inside\Permission\Facades\Role::getHumanName($array['name']);
            }

            $array['locked'] = in_array(
                $array['name'],
                array_merge(
                    ['authenticated', 'super_administrator', 'administrator', 'webmaster'],
                    config('permission.locked', [])
                )
            );

            if (get_class($node) == Role::class) {
                $array['users_count'] = $node->users()->where('status', true)->count();
            }

            $array['type'] = $type;

            $array['children'] = [];

            $this->node($array['children'], $node);

            // Filter workflow roles
            // TODO: Found a cleaner and global solution for filter internal roles
            if (strpos($array['name'], 'workflow-') === 0) {
                continue;
            }

            $tree[] = $array;
        }
    }

    /**
     * Handle role assignment
     */
    public function handleRole(Role $role, User $user, ?Content $model = null): void
    {
        if ($this->checkRoleCondition($role, $model)) {
            $user->assignRole($role);
        } else {
            $user->revokeRole($role);
        }

        Role::where('pid', $role->id)->whereNull('type')
          ->each(fn ($child) => $this->handleRole($child, $user));
    }

    protected function checkRoleConditionForReference(Content $model, string $reference, string $condition): bool
    {
        $model = $model->load($reference);
        $userValues = collect($model->{$reference} ?? [])->pluck('uuid_host')->filter();

        $negateCondition = false;

        if (Str::startsWith($condition, 'not:')) {
            $negateCondition = true;
            $condition = Str::after($condition, 'not:');
        }

        $conditionValues = collect(explode('&', $condition));

        if ($conditionValues->contains('any')) {
            return $negateCondition ? $userValues->isEmpty() : $userValues->isNotEmpty();
        }

        if ($negateCondition) {
            return $userValues->every(fn ($value) => ! $conditionValues->contains($value));
        }

        return $userValues->some(fn ($value) => $conditionValues->contains($value));
    }

    protected function checkRoleCondition(Role $role, ?Content $model = null): bool
    {
        if (! $role->condition || ! $model) {
            return true;
        }

        // All conditions have to true
        $conditions = explode('|', $role->condition);
        foreach ($conditions as $condition) {
            if (! Str::contains($condition, ':')) {
                continue;
            }

            [$fieldName, $condition] = explode(':', $condition, 2);

            if (method_exists($model, Str::camel($fieldName))) {
                $isValid = $this->checkRoleConditionForReference($model, Str::camel($fieldName), $condition);
            } elseif (isset($model->{$fieldName})) {
                $isValid = $model->{$fieldName} == $condition;
            } else {
                $isValid = false;
            }

            if (! $isValid) {
                return false;
            }
        }

        return true;
    }

    /**
     * Get human role name from machine name
     *
     * @param string $machineName
     * @param string|null $locale
     * @return string
     */
    public function getHumanName(string $machineName, ?string $locale = null): string
    {
        if ($locale === null) {
            $locale = app()->getLocale();
        }

        return Lang::get('roles.'.$machineName, [], $locale) === 'roles.'.$machineName ? $machineName
      : Lang::get('roles.'.$machineName, [], $locale);
    }

    /**
     * Return user roles
     *
     * @param User $user
     * @return array
     */
    public function listUserRoles(User $user): array
    {
        return $user->roles->where('type', '!=', 'group')->pluck('name')->toArray();
    }

    /**
     * Return user roles ids
     *
     * @param User $user
     * @return array
     */
    public function listUserRoleIds(User $user): array
    {
        return $user->roles->pluck('id')->toArray();
    }

    /**
     * List users attached to a role
     */
    public function listRoleUsers(
    int | array | string $ids,
    int $limit = null,
    string $search = null,
    bool $inRole = false,
    int $page = 1,
    bool $excludeCurrentUser = false,
    array $filters = null,
  ): array | Paginator {
        if (is_int($ids)) {
            $ids = [$ids];
        } elseif (! is_array($ids)) {
            $ids = explode(',', $ids);
        }

        $roles = Role::whereIn('id', $ids)->get();

        if ($roles->isEmpty()) {
            return [];
        }

        $ids = $roles->pluck('id');

        $query = Users::orderBy('lastname')->orderBy('firstname');

        if ($excludeCurrentUser) {
            $currentUser = Auth::user();

            $query->where('uuid', '<>', $currentUser->uuid);
        }

        if ($search) {
            $query->where(function ($query) use ($search) {
                $query->whereLike(DB::raw("concat(COALESCE(firstname, ''), ' ', COALESCE(lastname, ''))"), '%'.$search.'%');
                $query->orWhereLike(DB::raw("concat(COALESCE(lastname, ''), ' ', COALESCE(firstname, ''))"), '%'.$search.'%');
            });
        }

        if ($filters && isset($filters['$or'])) {
            $conditions = collect($filters['$or']);
            $query->where(function ($query) use ($conditions) {
                $conditions->each(function ($condition) use ($query) {
                    $query->orWhere(function ($query) use ($condition) {
                        collect($condition)->each(function ($value, $key) use ($query) {
                            if (str_contains($key, ':gte')) {
                                $actualKey = explode(':', $key)[0];
                                $query->where($actualKey, '>=', $value);
                            } else {
                                $query->where($key, $value);
                            }
                        });
                    });
                });
            });
        }

        if ($inRole) {
            $query->join('inside_users_roles', 'inside_content_users.uuid', '=', 'inside_users_roles.user_uuid')
        ->whereIn(
            'inside_users_roles.role_id',
            $ids
        );
        }

        if ($limit) {
            $result = $query->paginate($limit, ['uuid', 'email', 'firstname', 'lastname', 'image'], 'page', $page);
        } else {
            $count = 0;

            foreach ($roles as $role) {
                $count += $role->users->count();

                if ($excludeCurrentUser) {
                    if ($role->users()->where('uuid', $currentUser->uuid)->exists()) {
                        $count -= 1;
                    }
                }
            }

            $result = [
                'count' => $count,
                'total' => $query->count(),
                'data' => $query->get(['uuid', 'email', 'firstname', 'lastname', 'image']),
            ];
        }

        foreach ($result instanceof Paginator ? $result->items() : $result['data'] as &$user) {
            if (empty($user['image'])) {
                $user['image'] = null;
            } else {
                $value = $user['image'];
                $model = Users::query()->find($user['uuid']);
                $original = $value;
                $value = protected_file_url($model, 'image');
                // we add by default a webp file for the main image too so the front can use an optimized image
                // we add a little trick because it's not a real existing style, just the main image as webp
                /** @var array $value */
                $value = [
                    'main' => $value,
                    'webp' => [
                        'main' => protected_file_url($model, 'image', true, 'main-webp', true),
                    ],
                ];

                // Add styled image
                $options = Schema::getFieldOptions(class_to_type($model), 'image');
                if (isset($options['image_styles']) && ! empty($options['image_styles'])) {
                    $styles = ImageStyles::find($options['image_styles']);
                    foreach ($styles as $style) {
                        $value[Str::slug($style->title)] = protected_file_url($model, 'image', true, $style->title);
                        $value['webp'][Str::slug($style->title)] = protected_file_url($model, 'image', true, $style->title, true);
                    }
                }

                $user['image'] = $value;
            }
            $user['hasRole'] = false;

            foreach ($roles as $role) {
                if ($role->users()->where('uuid', $user->uuid)->exists()) {
                    $user['hasRole'] = true;
                    break;
                }
            }
        }

        return $result;
    }

    /**
     * Assign users to specific role
     *
     * @param int $roleId
     * @param array $uuids
     * @return void
     */
    public function assignUsersToRole(int $roleId, array $uuids): void
    {
        $assign = [];

        foreach ($uuids as $uuid) {
            $assign[] = [
                'role_id' => $roleId,
                'user_uuid' => $uuid,
            ];
        }

        try {
            DB::table('inside_users_roles')->insert($assign);
        } catch (QueryException $e) {
            // Do nothing, already attached to this role
        }
        $role = Role::find($roleId);
        $userEmail = Auth::user()?->email ?? 'system';
        Log::info("[PERMISSION] role assignation operated by $userEmail");
        foreach ($uuids as $uuid) {
            $user = User::find($uuid);
            Log::info("[PERMISSION] assigning role $role->name to user $user?->email");
            RoleAttachedEvent::dispatch($user, $role);
        }
    }

    /**
     * Revoke users to specific role
     *
     * @param int $roleId
     * @param array $uuids
     * @return void
     */
    public function revokeUsersToRole(int $roleId, array $uuids): void
    {
        DB::table('inside_users_roles')->whereIn('user_uuid', $uuids)->where('role_id', $roleId)->delete();
        $role = Role::find($roleId);
        $userEmail = Auth::user()?->email ?? 'system';
        Log::info("[PERMISSION] role revokation changed by $userEmail");
        foreach ($uuids as $uuid) {
            $user = User::find($uuid);
            Log::info("[PERMISSION] revoking role $role->name to user $user?->email");
            RoleDetachedEvent::dispatch($user, $role);
        }
    }

    /**
     * @param int $roleId
     * @return int
     */
    public function getUserCountForRole(int $roleId): int
    {
        $cacheKey = 'inside_permission.roles.count.'.$roleId;
        if (! Cache::has($cacheKey)) {
            Cache::forever($cacheKey, Role::findOrFail($roleId)->users()->whereStatus(true)->count());
        }

        return Cache::get($cacheKey);
    }

    /**
     * Create a role
     *
     * @param array $data
     * @return Role
     */
    public function create(array $data): Role
    {
        return Role::create($data);
    }

    /**
     * Update a role
     *
     * @param int $id
     * @param array $data
     * @return Role
     */
    public function update(int $id, array $data): Role
    {
        $role = Role::find($id);

        if ($role) {
            $role->update($data);
        }

        return $role;
    }

    /**
     * Delete a role
     *
     * @param int $id
     * @return void
     * @throws Exception
     */
    public function delete(int $id)
    {
        $role = Role::find($id);

        if ($role) {
            $parents = Role::where('pid', $id)->whereNull('type')->get();

            foreach ($parents as $parent) {
                $parent->pid = null;
                $parent->save();
            }

            $role->delete();
        }
    }

    /**
     * Create a role
     *
     * @param array $data
     * @return RoleCategory
     */
    public function createCategory(array $data): RoleCategory
    {
        return RoleCategory::create($data);
    }

    /**
     * Update a role
     *
     * @param int $id
     * @param array $data
     * @return RoleCategory
     */
    public function updateCategory(int $id, array $data): RoleCategory
    {
        $roleCategory = RoleCategory::find($id);

        if ($roleCategory) {
            $roleCategory->update($data);
        }

        return $roleCategory;
    }

    /**
     * Delete a role category
     *
     * @param int $id
     * @return void
     * @throws Exception
     */
    public function deleteCategory(int $id)
    {
        $roleCategory = RoleCategory::find($id);

        if ($roleCategory) {
            $roles = Role::where('role_category_id', $id)->whereNull('type')->get();
            $parents = RoleCategory::where('pid', $id)->get();

            foreach ($roles as $role) {
                $role->role_category_id = null;
                $role->save();
            }

            foreach ($parents as $parent) {
                $parent->pid = null;
                $parent->save();
            }

            $roleCategory->delete();
        }
    }
}
