<?php

namespace Inside\Permission\Services;

use Exception;
use Illuminate\Database\QueryException;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Inside\Content\Exceptions\ModelSchemaNotFoundException;
use Inside\Content\Facades\Schema;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Models\PermissionSchema;

/**
 * Inside helper permission service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PermissionHelperService
{
    /**
     * The unique instance
     *
     * @static
     */
    private static ?PermissionHelperService $instance = null;

    /**
     * @var array
     */
    private array $permissible = [];

    /**
     * Create a new Inside instance
     */
    public function __construct()
    {
        foreach (Schema::getContentTypes() as $model) {
            try {
                $modelOptions = Schema::getModelOptions($model);
            } catch (ModelSchemaNotFoundException) {
                continue;
            }
            if (array_key_exists('permissible', $modelOptions) && $modelOptions['permissible']) {
                $this->permissible[] = type_to_class($model);
            }
        }
    }

    /**
     * Get the instance if already exist
     */
    public static function getInstance(): self
    {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Return permissible models
     *
     * @return array
     */
    public function getPermissible(): array
    {
        return $this->permissible;
    }

    /**
     * Attribute specific permissions to franchised spaces folders $uuids to let $roles read it !
     *
     * @param string $uuid
     * @param array $roleToBeGrantedIds
     * @param array $roleToBeRejectedIds
     * @throws Exception
     */
    protected function setPermissions(string $uuid, array $roleToBeGrantedIds, array $roleToBeRejectedIds = []): void
    {
        $permissionSchema = [
            'authorizable_type' => type_to_class('franchised_spaces_folders'),
            'authorizable_uuid' => $uuid,
            'action' => 'read',
            'invert' => false,
            'children' => true,
        ];
        // grant
        foreach ($roleToBeGrantedIds as $roleId) {
            $permissionSchemaID = $this->getOrCreatePermissionSchema($permissionSchema);

            $this->grantPermissionSchemaToRoleId($permissionSchemaID, $roleId);
            Permission::buildPermissionForRole($roleId);
        }
        // reject
        $permissionSchema['invert'] = true;
        foreach ($roleToBeRejectedIds as $roleId) {
            $permissionSchemaID = $this->getOrCreatePermissionSchema($permissionSchema);

            $this->grantPermissionSchemaToRoleId($permissionSchemaID, $roleId);
            Permission::buildPermissionForRole($roleId);
        }
    }

    /**
     * get or create a permission schema using $schema config
     *
     * @param array $schema
     * @return int
     */
    public function getOrCreatePermissionSchema(array $schema): int
    {
        $permissionSchema = DB::table('inside_permissions_schema')->where($schema)->first();

        if (! is_null($permissionSchema)) {
            return $permissionSchema->id; // @phpstan-ignore-line
        }

        return DB::table('inside_permissions_schema')->insertGetId($schema);
    }

    /**
     * Grant permission schema $schemaId to $roleId
     *
     * @param int $schemaId
     * @param int $roleId
     */
    protected function grantPermissionSchemaToRoleId(int $schemaId, int $roleId): void
    {
        try {
            DB::table('inside_roles_permissions_schema')->insert(
                [
                    'role_id' => $roleId,
                    'permission_schema_id' => $schemaId,
                    'is_content_specific' => true,
                ]
            );
        } catch (QueryException) {
            // Do nothing
            Log::error('[FranchisedSpacesListeners] Failed to grant permission schema <'.$schemaId.'> to role <'.$roleId.'>');
        }
    }
}
