<?php

namespace Inside\Permission\Scopes;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Scope;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Services\PermissionHelperService;

/**
 * Scope to filter read permissions on content load
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AllowedScope implements Scope
{
    public function apply(Builder $builder, Model $model): void
    {
        if (! Permission::isAllowedScopeEnable() || php_sapi_name() === 'cli') {
            return;
        }

        if (Permission::isSystemV2Enabled()) {
            $this->v2($builder, $model);
        } else {
            $this->v1($builder, $model);
        }
    }

    protected function v1(Builder $builder, Model $model): void
    {
        $helper = PermissionHelperService::getInstance();
        $permissibleContentClasses = $helper->getPermissible();
        if (php_sapi_name() === 'cli' || ! in_array(get_class($model), $permissibleContentClasses)) {
            return;
        }

        $user = Auth::user();
        if ($user === null || $user->permission === null) {
            // If not authenticated, just return nothing.
            $builder->whereNull($model->getTable().'.uuid');

            return;
        }

        // If super admin, we don't need any filter Or Client Application
        if (! $user instanceof User || Permission::isSuperAdmin($user)) {
            return;
        }

        $table = class_to_table(get_class($model));

        if (! $table) {
            return;
        }

        // Special for scope, we need to join permission table to filter directly on read permissions
        $applyPermissionScope = true;
        $customApply = config('permission.apply_permission_scope');

        if ($customApply && is_callable($customApply)) {
            $applyPermissionScope = $customApply($user, $model);
        } elseif (is_array($customApply)) {
            foreach ($customApply as $apply) {
                if ($apply && is_callable($apply)) {
                    $applyPermissionScope = $apply($user, $model);
                }
            }
        }

        if ($applyPermissionScope) {
            $builder->whereIn("$table.uuid", fn ($query) => $query
                ->select('uuid')
                ->from('inside_permissions')
                ->where('type', table_to_class($table))
                ->where('inside_permissions.action', 'read')
                ->whereIn('inside_permissions.role_id', $user->permission->getRoleIds())
                ->whereNotNull('uuid')
            );
        }

        $customScope = config('permission.global_scope', false);
        if ($customScope && is_callable($customScope)) {
            $customScope($builder, $user, $model);
        } elseif (is_array($customScope)) {
            foreach ($customScope as $scope) {
                if ($scope && is_callable($scope)) {
                    $scope($builder, $user, $model);
                }
            }
        }
    }

    protected function v2(Builder $builder, Model $model): void
    {
        /** @var User|null $user */
        $user = Auth::user();

        if (! $user) {
            $builder->whereNull($model->getTable().'.uuid');

            return;
        }

        if ($user->isSuperAdmin()) {
            return;
        }

        if (! $model instanceof Content) {
            return;
        }

        if (! $model::isPermissible()) {
            return;
        }

        $builder->where(function ($query) use ($user, $model) {
            if ($user->hasContentTypePrivilegeTo(CapabilityEnum::UPDATE, $model)) {
                match ($model::isCategorizable()) {
                    true => $user->getRolePrivilegesService()->applyCategorizableContentAccessRestriction($query, [CapabilityEnum::READ, CapabilityEnum::UPDATE]),
                    false => $user->getRolePrivilegesService()->applyContentUpdateRestriction($query, $model),
                };
            }

            $query->orWhere(function ($orQuery) use ($user, $model) {
                match ($model::isCategorizable()) {
                    true => $user->getRolePrivilegesService()->applyCategorizableContentAccessRestriction($orQuery),
                    false => $user->getRolePrivilegesService()->applyContentViewRestriction($orQuery, $model),
                };
            });
        });
    }
}
