<?php

namespace Inside\Permission\Providers;

use Illuminate\Support\Facades\Gate;
use Inside\Authentication\Models\User;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Events\ContentDeletedEvent;
use Inside\Content\Events\ContentFullyInsertedEvent;
use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Content\Models\Content;
use Inside\Facades\Inside;
use Inside\Permission\Console\CleanupCommand;
use Inside\Permission\Console\FixPermissionSchemaCommand;
use Inside\Permission\Console\PermissionsRebuildCommand;
use Inside\Permission\Console\PermissionsV2DisableCommand;
use Inside\Permission\Console\PermissionsV2EnableCommand;
use Inside\Permission\Console\PermissionsV2InitializeCommand;
use Inside\Permission\Console\PermissionsV2MigrateCommand;
use Inside\Permission\Console\PermissionsV2RebuildCommand;
use Inside\Permission\Console\PermissionsV2RequirementCommand;
use Inside\Permission\Events\RoleCreatedEvent;
use Inside\Permission\Events\RoleDeletedEvent;
use Inside\Permission\Events\RoleUpdatedEvent;
use Inside\Permission\Exodus\Listeners\ComputeRoleRestrictionsOnSavedContent;
use Inside\Permission\Exodus\Listeners\CreatePrivilegesOnNewCategorizableContent;
use Inside\Permission\Exodus\Listeners\RemovePrivilegesOnDeletedCategorizableContent;
use Inside\Permission\Exodus\Listeners\RemovePrivilegesOnDeletedContent;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Http\Middlewares\AddRoleToQueryMiddleware;
use Inside\Permission\Http\Middlewares\ContentFormMiddleware;
use Inside\Permission\Http\Middlewares\LayoutPermissionMiddleware;
use Inside\Permission\Http\Middlewares\RoleMiddleware;
use Inside\Permission\Http\Middlewares\SuperAdminMiddleware;
use Inside\Permission\Listeners\AttachRoleToUserListener;
use Inside\Permission\Listeners\AttachUserToRoleListener;
use Inside\Permission\Listeners\AutoAddCreatePermissionOnSpecialContents;
use Inside\Permission\Listeners\DeletePermissionListener;
use Inside\Permission\Listeners\UpdatePermissionListener;
use Inside\Permission\Listeners\UserDisabledListener;
use Inside\Permission\Policies\ContentPolicy;
use Inside\Permission\Services\PermissionSchemaService;
use Inside\Permission\Services\PermissionService;
use Inside\Permission\Services\RoleService;
use Inside\Support\EventServiceProvider;
use Inside\Support\Str;
use Inside\User\Events\UserDisabledEvent;

/**
 * Inside Permission service.
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PermissionServiceProvider extends EventServiceProvider
{
    protected array $listen = [
        ContentFullyInsertedEvent::class => [
            AttachRoleToUserListener::class,
        ],
        ContentCreatedEvent::class       => [
            // V1 to delete
            UpdatePermissionListener::class,
            AutoAddCreatePermissionOnSpecialContents::class,
            // V2 to keep
            CreatePrivilegesOnNewCategorizableContent::class,
            ComputeRoleRestrictionsOnSavedContent::class,
            AttachRoleToUserListener::class,
        ],
        ContentUpdatedEvent::class       => [
            // V1 to delete
            UpdatePermissionListener::class,
            // V2 to keep
            ComputeRoleRestrictionsOnSavedContent::class,
            AttachRoleToUserListener::class,
        ],
        ContentDeletedEvent::class       => [
            // V1 to delete
            DeletePermissionListener::class,
            // V2 to keep
            RemovePrivilegesOnDeletedContent::class,
            RemovePrivilegesOnDeletedCategorizableContent::class,
        ],
        RoleCreatedEvent::class          => [
            AttachUserToRoleListener::class,
        ],
        RoleUpdatedEvent::class          => [
            AttachUserToRoleListener::class,
        ],
        RoleDeletedEvent::class          => [
            AttachUserToRoleListener::class,
        ],
        UserDisabledEvent::class         => [
            UserDisabledListener::class,
        ],
    ];

    // V1 to delete
    protected array $commands = [
        PermissionsRebuildCommand::class,
        CleanupCommand::class,
        FixPermissionSchemaCommand::class,
        PermissionsV2InitializeCommand::class,
        PermissionsV2EnableCommand::class,
        PermissionsV2DisableCommand::class,
        PermissionsV2MigrateCommand::class,
        PermissionsV2RequirementCommand::class,
        PermissionsV2RebuildCommand::class,
    ];

    public array $bindings = [
        'role' => RoleService::class,
    ];

    public array $singletons = [
        'permission.schema' => PermissionSchemaService::class,
        'permission' => PermissionService::class,
    ];

    /**
     * Bootstrap the application services.
     *
     * @return void
     */
    public function boot(): void
    {
        parent::boot();

        $this->prepareGateForBackoffice();

        Gate::policy(Content::class, ContentPolicy::class);
    }

    /**
     * Register the application services.
     *
     * @return void
     */
    public function register()
    {
        $this->mergeConfigFrom(__DIR__.'/../../config/permission.php', 'permission');

        $this->app->bind(
            'role',
            function () {
                return new RoleService();
            }
        );

        $this->app->routeMiddleware(['super_admin' => SuperAdminMiddleware::class]);

        $this->app->middleware(
            [
                AddRoleToQueryMiddleware::class,
                ContentFormMiddleware::class,
                LayoutPermissionMiddleware::class,
            ]
        );
        $this->app->routeMiddleware(['auth.role' => RoleMiddleware::class]);
    }

    /**
     * prepare gate authorizations
     */
    protected function prepareGateForBackoffice(): void
    {
        Gate::before(function ($user) {
            if ($user instanceof User && $user->isSuperAdmin()) {
                return true;
            }
        });

        foreach (Inside::getAllBackofficeEntries() as $backofficeEntry) {
            Gate::define('access_'.$backofficeEntry, function (User $user) use ($backofficeEntry) {
                return Permission::userCanAccessBackofficeEntry(Str::lower(trim($backofficeEntry)), $user->permission);
            });
        }
    }
}
