<?php

namespace Inside\Permission\Policies;

use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Permission\Facades\Permission;

/**
 * Define content policy
 *d
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class ContentPolicy
{
    /**
     * Override policy. Bypass for super administrator
     * @return void|bool
     */
    public function before(User $user)
    {
        if ($user->isSuperAdmin()) {
            return true;
        }
    }

    /**
     * Can this user view this type of content
     *
     * @param User $user
     * @param Content $content
     *
     * @return bool
     */
    public function view(User $user, Content $content): bool
    {
        return (Permission::isSystemV2Enabled())
            ? Permission::can('read', $content, $user)
            : Permission::allowed('read', $content->content_type, $content->uuid, $user);
    }

    /**
     * Can we create a new content ?
     *
     * @param User    $user    use that want to store a new content
     * @param Content $content content model
     *
     * @return bool
     */
    public function create($user, Content $content): bool
    {
        return (Permission::isSystemV2Enabled())
            ? Permission::can('create', $content, $user)
            : Permission::allowed('create', $content->content_type, $content->uuid, $user);
    }

    /**
     * Can we edit this content ?
     *
     * @param User    $user    user does want to edit
     * @param Content $content content to update
     *
     * @return bool
     */
    public function update($user, Content $content): bool
    {
        return (Permission::isSystemV2Enabled())
            ? Permission::can('update', $content, $user)
            : Permission::allowed('update', $content->content_type, $content->uuid, $user);
    }

    /**
     * Can we delete this specific content ?
     *
     * @param User    $user    user that wants to delete this content
     * @param Content $content the content to delete
     *
     * @return bool
     */
    public function delete($user, Content $content): bool
    {
        return (Permission::isSystemV2Enabled())
            ? Permission::can('delete', $content, $user)
            : Permission::allowed('delete', $content->content_type, $content->uuid, $user);
    }
}
