<?php

namespace Inside\Permission\Models\Traits;

use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Inside\Permission\Events\RoleAttachedEvent;
use Inside\Permission\Events\RoleDetachedEvent;
use Inside\Permission\Models\Role;

/**
 * Trait HasRoles
 *
 * Add roles to the model & its useful functions
 */
trait HasRoles
{
    /**
     * Get all user roles.
     *
     * @return BelongsToMany
     */
    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(
            Role::class,
            'inside_users_roles',
            'user_uuid',
            'role_id'
        );
    }

    /**
     * Determine if the model has any of the given role(s).
     *
     * @param  string|Role|array|int  $roles
     * @return bool
     */
    public function hasAnyRole($roles): bool
    {
        if (is_string($roles) && false !== strpos($roles, '|')) {
            $roles = $this->convertPipeToArray($roles);
        }
        if (is_string($roles)) {
            return $this->roles->contains('name', $roles);
        }
        if (is_int($roles)) {
            return $this->roles->contains('id', $roles);
        }
        if ($roles instanceof Role) {
            return $this->roles->contains('id', $roles->id);
        }
        if (is_array($roles)) {
            foreach ($roles as $role) {
                if ($this->hasRole($role)) {
                    return true;
                }
            }

            return false;
        }

        return $roles->intersect($this->roles)->isNotEmpty();
    }

    /**
     * Determine if the model has ALL of the given role(s).
     *
     * @param  string|Role|array|int  $roles
     * @return bool
     */
    public function hasAllRoles($roles): bool
    {
        if (is_string($roles) && false !== strpos($roles, '|')) {
            $roles = $this->convertPipeToArray($roles);
        }
        if (is_string($roles)) {
            return $this->roles->contains('name', $roles);
        }
        if ($roles instanceof Role) {
            return $this->roles->contains('id', $roles->id);
        }
        $roles = collect()->make($roles)->map(function ($role) {
            return $role instanceof Role ? $role->name : $role;
        });

        return $roles->intersect($this->getRoleNames()) == $roles;
    }

    /**
     * Get Roles names
     *
     * @return Collection
     */
    public function getRoleNames(): Collection
    {
        return $this->roles->pluck('name');
    }

    /**
     * Get Roles ids
     *
     * @return Collection
     */
    public function getRoleIds(): Collection
    {
        return $this->roles->pluck('id');
    }

    protected function convertPipeToArray(string $pipeString): array|string
    {
        $pipeString = trim($pipeString);
        if (strlen($pipeString) <= 2) {
            return $pipeString;
        }
        $quoteCharacter = substr($pipeString, 0, 1);
        $endCharacter = substr($quoteCharacter, -1, 1);
        if ($quoteCharacter !== $endCharacter) {
            return explode('|', $pipeString);
        }
        if (! in_array($quoteCharacter, ["'", '"'])) {
            return explode('|', $pipeString);
        }

        return explode('|', trim($pipeString, $quoteCharacter));
    }

    /**
     * Does we have one or any of $roles ?
     *
     * @param  mixed  $role
     * @return bool
     */
    public function hasRole($role): bool
    {
        $role = $this->getRole($role);

        return $this->roles->contains($role);
    }

    /**
     * Find a role
     *
     * @param  mixed  $role
     * @return Role
     */
    public function getRole($role): ?Role
    {
        if (is_string($role)) {
            $role = Role::where(['name' => $role])->whereNull('type')->first();
        } elseif (is_int($role)) {
            $role = Role::find($role);
        }

        return $role;
    }

    /**
     * Assign role to user
     *
     * @param  mixed  $role
     * @return void
     */
    public function assignRole($role)
    {
        $userEmail = Auth::user()?->email ?? 'system';
        $role = $this->getRole($role);
        try {
            $this->roles()->attach($role);
            Log::info("[PERMISSION] assigning role $role?->name to user $this->email by $userEmail");
            event(new RoleAttachedEvent($this, $role));
        } catch (\Exception $e) {
            //
        }
    }

    /**
     * Detach role to user
     *
     * @param  mixed  $role
     * @return void
     */
    public function revokeRole($role)
    {
        $userEmail = Auth::user()?->email ?? 'system';
        $role = $this->getRole($role);
        $this->roles()->detach($role);
        Log::info("[PERMISSION] revoking role $role?->name to user $this->email by $userEmail");
        event(new RoleDetachedEvent($this, $role));
    }

    public function revokeAllRoles(): void
    {
        foreach ($this->roles as $role) {
            $this->revokeRole($role);
        }
    }
}
