<?php

namespace Inside\Permission\Listeners;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Inside\Content\Events\ContentCreatedEvent;
use Inside\Content\Events\ContentUpdatedEvent;
use Inside\Content\Models\Content;
use Inside\Content\Models\Model;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Services\PermissionHelperService;

/**
 * Attach schema to role listener
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class UpdatePermissionListener
{
    public function handle(ContentCreatedEvent|ContentUpdatedEvent $event): void
    {
        $model = $event->model;
        $helper = PermissionHelperService::getInstance();
        $permissibles = $helper->getPermissible();

        if (! in_array(get_class($model), $permissibles)) {
            return;
        }

        if (PermissionSchema::hasContentSpecificPermission($model)) {
            return;
        }

        $roleIds = DB::table('inside_roles')->get()->pluck('id');

        /** @var Model $modelInfos */
        $modelInfos = Model::where('class', get_class($model))->firstOrFail();

        $newPermissions = [];
        $removePermissions = [];
        foreach ($roleIds as $roleId) {
            // Global allowed action > Direct allowed actions
            $direct = collect(PermissionSchema::getActionsForRoles([$roleId], $modelInfos, $model))->filter(
                function ($value) {
                    return ! is_null($value);
                }
            );
            $global = collect(PermissionSchema::getActionsForRoles([$roleId], $modelInfos))->filter(function ($value) {
                return ! is_null($value);
            });

            $inherited =
                collect($this->allowedActionsByParent($roleId, $modelInfos, $model))->filter(function ($value) {
                    return ! is_null($value);
                });

            $actions = $direct->merge($global)->merge($inherited);

            foreach ($actions as $action => $schema) {
                if ($schema !== null) {
                    if ($schema['invert'] == 0) {
                        // This action is allowed for this role !
                        $permission = [
                            'role_id' => $roleId,
                            'action'  => $action,
                            'uuid'    => $model->uuid,
                            'type'    => $modelInfos->model, // magic get class name
                        ];
                        if (! DB::table('inside_permissions')->where($permission)->exists()) {
                            $newPermissions[] = $permission;
                        }
                    } else {
                        if ($schema['invert'] == 1) {
                            // This action is NOT allowed for this role !
                            $permission = [
                                'role_id' => $roleId,
                                'action'  => $action,
                                'uuid'    => $model->uuid,
                                'type'    => $modelInfos->model,  // magic get class name
                            ];
                            if (DB::table('inside_permissions')->where($permission)->exists()) {
                                $removePermissions[] = $permission;
                            }
                        }
                    }
                }
            }
        }
        try {
            DB::table('inside_permissions')->insert($newPermissions);
            foreach ($removePermissions as $permission) {
                DB::table('inside_permissions')->where($permission)->delete();
            }
        } catch (Exception $e) {
        }
        // Remove Permission by inheritance
        Permission::removePermissionByInheritance($model);

        Permission::removeChildrenPermission($model);
    }

    /**
     * Get allowed actions by parent !
     *
     * @param int $roleId
     * @param Model $modelInfos
     * @param mixed $content
     * @return array
     */
    protected function allowedActionsByParent(int $roleId, Model $modelInfos, $content): array
    {
        $type = class_to_type($content);
        // Only get parents of type $modelInfos
        $parentsByTypes = $content->getParentsIfExist(['content_type' => $type]);
        if (! empty($parentsByTypes[$type])) {
            return PermissionSchema::getActionsForRoles([$roleId], $modelInfos, Arr::last($parentsByTypes[$type]));
        }

        if (class_to_type($content) === 'main_menus') {
            // On most instances, main menus do not have NULL permission and there is no way to set them...
            // Main menu items should always be readable by default
            return [
                'read'   => [
                    'action'            => 'read',
                    'children'          => true,
                    'invert'            => false,
                    'authorizable_type' => get_class($content),
                    'authorizable_uuid' => null,
                ],
                'create' => null,
                'update' => null,
                'delete' => null,
            ];
        }

        return [];
    }

    /**
     * Get recursively parent schemas
     */
    protected function getRelationSchemas(Content $relation): array
    {
        $schemas = [];

        $referenceSchema = DB::table('inside_permissions_schema')->where([
            'authorizable_uuid' => $relation->relationUuid,
            'children'          => true,
        ])->get()->toArray();

        if (count($referenceSchema)) {
            return array_merge($schemas, $referenceSchema);
        }

        if ($relation->pid) {
            $parent = call_user_func(get_class($relation).'::find', $relation->pid);

            if ($parent) {
                $schemas = $this->getRelationSchemas($parent);
            }
        }

        return $schemas;
    }
}
