<?php

namespace Inside\Permission\Http\Middlewares;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\Role;
use Symfony\Component\HttpFoundation\Response;

/**
 * Add role to current and login queries
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class AddRoleToQueryMiddleware
{
    /**
     * The path where permission are injected
     *
     * @var array
     */
    private $paths = [
        'api/v1/authentication/current',
        'api/v1/authentication/login',
    ];

    /**
     * Handle an incoming request.
     *
     * @param Request $request
     * @param Closure                 $next
     *
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        $response = $next($request);
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        // Check path and user
        $path = $request->path();
        $user = Auth::user();

        if (! in_array($path, $this->paths) || ! $user) {
            return $response;
        }
        if ($response instanceof Response && $response->getStatusCode() != 200) {
            return $response;
        }

        // Attach roles to data
        /** @var array $content */
        $content = json_decode_response($response);
        if (! array_key_exists('data', $content)) {
            return $response;
        }
        $userPermission = Permission::user($user);

        $content['data']['permissions'] = Cache::remember(
            md5('inside.permission.'.$userPermission->uuid.'.informations'),
            10, // Safe navigation for 10 minutes ( note: rememberforever could have been a solution
            // but is quite pain in the ass : lots of event could trigger a clear cache )
            function () use ($userPermission) {
                return [
                    'roles'  => Role::listUserRoles($userPermission),
                    'add'    => Permission::getAllowedCreationTypesForUser($userPermission),
                    'custom' => Permission::customList($userPermission),
                    'backoffice_entries' => Permission::backofficeAccessibleEntries($userPermission),
                ];
            }
        );

        set_response($response, $content);

        return $response;
    }
}
