<?php

namespace Inside\Permission\Http\Controllers;

use Exception;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Inside\Authentication\Models\User;
use Inside\Content\Facades\ContentCache;
use Inside\Permission\Facades\Permission;
use Inside\Permission\Facades\PermissionSchema;
use Inside\Permission\Facades\Role;
use Laravel\Lumen\Routing\Controller;

/**
 * Role Controller
 *
 * @category Class
 * @author   Maecia <technique@maecia.com>
 * @license  http://www.gnu.org/copyleft/gpl.html GNU General Public License
 * @link     http://www.maecia.com/
 */
class PermissionController extends Controller
{
    /**
     * List role tree
     */
    public function index(int $roleId): JsonResponse
    {
        $tree = PermissionSchema::tree([$roleId]);

        return response()->json([
            'data' => $tree,
        ]);
    }

    /**
     * List user role tree
     */
    public function user(string $uuid): JsonResponse
    {
        $user = User::findOrFail($uuid);

        $roleIds = Role::listUserRoleIds($user);
        $roleNames = Role::listUserRoles($user);

        // Filter workflow roles
        foreach ($roleNames as $key => $role) {
            if (str_starts_with($role, 'workflow-')) {
                unset($roleNames[$key]);
            }
        }

        $tree = PermissionSchema::tree($roleIds);

        return response()->json([
            'data' => $tree, 'roles' => array_values($roleNames),
        ]);
    }

    /**
     * Create permissions schema
     * @throws Exception
     */
    public function post(Request $request, int $roleId): JsonResponse
    {
        $data = $request->only('schemas');

        foreach (list_languages() as $locale) {
            if ($locale === App::getLocale()) {
                continue;
            }
            $data['schemas'] = PermissionSchema::getExistingMenuSchema($roleId, $locale, $data['schemas']);
        }

        PermissionSchema::deletePermissionsSchemaForRole($roleId);

        foreach ($data['schemas'] as $schemas) {
            PermissionSchema::create($schemas, $roleId);
        }

        Permission::buildPermissionForRole($roleId);
        PermissionSchema::clean();

        ContentCache::forget([config('app.key', 'sid2'), 'all']);

        return response()->json([
            'data' => PermissionSchema::tree([$roleId]),
        ]);
    }
}
