<?php

namespace Inside\Permission\Exodus\Services\GrantDefaultPrivilege;

use Illuminate\Database\Query\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Inside\Content\Models\Content;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\ViewModels\CategorizableContentIndex;

class GrantDefaultCategorizableContentPrivileges
{
    protected Collection $relations;

    protected string $type;

    private function __construct(protected Content $content)
    {
        $this->relations = $content::getCategorizableRelations();
        $this->type = $content->getContentTypeAttribute();
    }

    public static function of(Content $content): bool
    {
        if (! $content::isPermissible() || ! $content::isCategorizable()) {
            return false;
        }

        (new static($content))->execute();

        return true;
    }

    protected function execute(): void
    {
        $capabilities = $this->isRecursive()
            ? $this->retrieveCapabilitiesThroughParentToApplyByRole()
            : $this->retrieveCapabilitiesToApplyByRole();

        CategorizableContentPrivilege::query()
            ->where('uuid_host', $this->content->uuid_host)
            ->each(function (CategorizableContentPrivilege $privilege) use ($capabilities) {
                $roles = $capabilities->get($privilege->capability_id, []);

                $privilege->roles()->sync($roles);
            });
    }

    /**
     * Get total of content of this type minus one
     * We need to subtract one because the content we are creating is not yet counted
     *
     * @return int
     */
    protected function total(): int
    {
        $relation = $this->relations->flip()->get($this->type);

        return $this->isRecursive()
            ? $this->content::query()->whereDoesntHave($relation)->count() - 1
            : $this->content::query()->count() - 1;
    }

    /**
     * @return bool
     */
    protected function isRecursive(): bool
    {
        return in_array($this->type, $this->relations->toArray());
    }

    /**
     * @return Collection
     */
    protected function retrieveCapabilitiesThroughParentToApplyByRole(): Collection
    {
        $relation = $this->relations->flip()->get($this->type);

        $parent = $this->content->{$relation}->first();

        if ($parent instanceof Content) {
            // If parent exist, we juste need to retrieve capabilities of this parent to apply to the new content, so we disable the filtering.
            return $this->retrieveCapabilitiesToApplyByRole([$parent->uuid_host], filtering: false);
        }

        // If no parent, it's mean the content is a root content. so At this point we need to retrieve all root content of this type to filter our query.
        $roots = $this->content::query()->whereDoesntHave($relation)->pluck('uuid_host');

        return $this->retrieveCapabilitiesToApplyByRole($roots->toArray());
    }

    /**
     * We retrieve every categorizable content privileges in relation with roles.
     * We filter roles that are in relations we all content of this type for each capability.
     * For example, we want to know if the role authenticated can view all news_categories, yes or no.
     * At the end we return capabilities that are allowed for each role.
     *
     * @param array $restriction An array of uuid_host to restrict the query
     * @param bool $filtering If it's true, that mean we want to compare the total of content of this type minus one with the count of content_privilege_id
     * @return Collection
     */
    protected function retrieveCapabilitiesToApplyByRole(array $restriction = [], bool $filtering = true): Collection
    {
        return DB::table(CategorizableContentPrivilege::PIVOT_ROLE_TABLE)
            ->join(CategorizableContentPrivilege::TABLE, CategorizableContentPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', '=', CategorizableContentPrivilege::TABLE.'.id')
            ->join(CategorizableContentIndex::TABLE, CategorizableContentPrivilege::TABLE.'.uuid_host', '=', CategorizableContentIndex::TABLE.'.uuid_host')
            ->select('role_id', 'capability_id', DB::raw('count(*) as count'))
            ->where('type', $this->content::class)
            ->when(! blank($restriction),
                fn (Builder $query) => $query->whereIn(CategorizableContentPrivilege::TABLE.'.uuid_host', $restriction)
            )
            ->groupBy('role_id', 'capability_id')
            ->get()
            ->map(fn ($row) => [
                'role_id' => $row->role_id,
                'capability_id' => $row->capability_id,
                'allowed' => $row->count === $this->total(),
            ])
            ->when($filtering,
                fn (Collection $privileges) => $privileges->filter(fn ($row) => $row['allowed'])
            )
            ->groupBy('capability_id')
            ->map(fn ($rows) => $rows->pluck('role_id')->toArray());
    }
}
