<?php

namespace Inside\Permission\Exodus\Services\AccessRestriction;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Contracts\AccessRestrictions;
use Inside\Permission\Exodus\Dto\CapabilityDto;
use Inside\Permission\Exodus\Dto\Indexes\ContentTranslationIndexDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Privileges\ContentPrivilege;
use Inside\Permission\Exodus\Models\ViewModels\ContentTranslationIndexes;
use Inside\Permission\Exodus\Services\AccessRestriction\Traits\HasCapababilities;

final class ContentAccessRestriction extends BaseAccessRestriction implements AccessRestrictions
{
    use HasCapababilities;

    private static array $capabilities = [
        CapabilityEnum::READ,
        CapabilityEnum::UPDATE,
        CapabilityEnum::DELETE,
    ];

    /**
     * @return Collection<ContentPrivilegeDto>
     */
    public static function expected(): Collection
    {
        return ContentTranslationIndexes::all()
            ->crossJoin(self::availableCapabilities())
            ->map(function (array $privileges) {
                /** @var ContentTranslationIndexes $index */
                /** @var CapabilityDto $capability */
                [$index, $capability] = $privileges;

                return ContentPrivilegeDto::from(
                    ContentTranslationIndexDto::fromModel($index),
                    $capability,
                );
            });
    }

    /**
     * @return Collection<ContentPrivilegeDto>
     */
    public static function availables(): Collection
    {
        return ContentPrivilege::query()->get()->collect()->map(
            fn (ContentPrivilege $privilege) => ContentPrivilegeDto::fromModel($privilege)
        );
    }

    public static function missing(): Collection
    {
        $availables = ContentPrivilege::query()->select('uuid')->get()->pluck('uuid')->toArray();

        return static::expected()->reject(function (ContentPrivilegeDto $dto) use ($availables) {
            return in_array($dto->getIndex()?->getTranslatableUuid(), $availables);
        });
    }

    public static function unnecessaries(): Collection
    {
        $expected = static::expected()->pluck('index.translatable_uuid')->toArray();

        return ContentPrivilege::query()->get()
            ->reject(fn (ContentPrivilege $privilege) => in_array($privilege->uuid, $expected))
            ->map(fn (ContentPrivilege $privilege) => ContentPrivilegeDto::fromModel($privilege));
    }
}
