<?php

namespace Inside\Permission\Exodus\Models;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Lang;
use Inside\Authentication\Models\User;
use Inside\Content\Models\Content;
use Inside\Permission\Exodus\Dto\Privileges\ContentTypePrivilegeDto;
use Inside\Permission\Exodus\Models\Privileges\BackofficePrivilege;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentSpecificPrivilege;
use Inside\Permission\Exodus\Models\Privileges\ContentTypePrivilege;
use Inside\Permission\Exodus\Models\Privileges\MenuPrivilege;
use Inside\Permission\Exodus\Models\Privileges\SpecificContentPrivilege;
use Inside\Permission\Exodus\Models\Privileges\StandardContentPrivilege;
use Inside\Permission\Exodus\Services\ComputeRestriction\ComputeRestrictionService;
use Inside\Permission\Exodus\Services\RolePrivilegesService;
use Inside\Permission\Models\RoleCategory;

/**
 * @property int $id
 * @property string $name
 * @property int $pid
 * @property int $role_category_id
 * @property bool $is_automatic
 * @property string $condition
 * @property int $weight
 * @property string $type
 * @property RoleCategory $category
 * @property Collection<User> $users
 * @property Collection<ContentTypePrivilege> $contentTypePrivileges
 * @property Collection<CategorizableContentPrivilege> $categorizablePrivileges
 * @property Collection<SpecificContentPrivilege> $specifiqueContentPrivileges
 * @property Collection<BackofficePrivilege> $backofficSectionPrivileges
 * @mixin Builder
 * @method Builder withoutDeprecatedRoles()
 * @method Builder withoutAuthenticated()
 * @method Builder withoutSuperAdministrator()
 */
class Role extends Model
{
    const TABLE = 'inside_roles';

    const AUTHENTICATED = 'authenticated';

    const SUPER_ADMINISTRATOR = 'super_administrator';

    /**
     * The table without the timestamps.
     * @var bool
     */
    public $timestamps = false;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = self::TABLE;

    /**
     * The attributes that are guarded.
     *
     * @var array
     */
    protected $guarded = ['id'];

    /**
     * Fillable properties
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'pid',
        'role_category_id',
        'is_automatic',
        'condition',
        'weight',
        'type',
    ];

    protected $appends = [
        'label',
    ];

    /**
     * The attributes that should be casted to native types.
     *
     * @var array
     */
    protected $casts = [
        'is_automatic' => 'boolean',
    ];

    /**
     * Remove deprecated roles from group and workflow from the query.
     * It could be removed in the future when all the deprecated roles will be removed.
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithoutDeprecatedRoles(Builder $query): Builder
    {
        return $query->whereNull('type');
    }

    public function scopeWithoutAuthenticated(Builder $query): Builder
    {
        return $query->where('name', '<>', self::AUTHENTICATED);
    }

    public function scopeWithoutSuperAdministrator(Builder $query): Builder
    {
        return $query->where('name', '<>', self::SUPER_ADMINISTRATOR);
    }

    public function getLabelAttribute(): string
    {
        $key = str($this->name)->start('roles.')->toString();

        return Lang::has($key) ? Lang::get($key) : $this->name;
    }

    public function isSuperAdmin(): bool
    {
        return $this->name === self::SUPER_ADMINISTRATOR;
    }

    public function isAuthenticated(): bool
    {
        return $this->name === self::AUTHENTICATED;
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function contentTypePrivileges(): BelongsToMany
    {
        return $this->belongsToMany(
            related: ContentTypePrivilege::class,
            table: ContentTypePrivilege::PIVOT_ROLE_TABLE,
            foreignPivotKey: 'role_id',
            relatedPivotKey: 'content_type_privilege_id'
        )->withPivot('pivot_id');
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function categorizableContentPrivileges(): BelongsToMany
    {
        return $this->belongsToMany(
            related: CategorizableContentPrivilege::class,
            table: CategorizableContentPrivilege::PIVOT_ROLE_TABLE,
            foreignPivotKey: 'role_id',
            relatedPivotKey: 'content_privilege_id'
        )->withPivot('pivot_id');
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function contentPrivileges(): BelongsToMany
    {
        return $this->belongsToMany(
            related: ContentPrivilege::class,
            table: ContentPrivilege::PIVOT_ROLE_TABLE,
            foreignPivotKey: 'role_id',
            relatedPivotKey: 'content_privilege_id'
        )->withPivot('pivot_id');
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function contentSpecificPrivileges(): BelongsToMany
    {
        return $this->belongsToMany(
            related: ContentSpecificPrivilege::class,
            table: ContentSpecificPrivilege::PIVOT_ROLE_TABLE,
            foreignPivotKey: 'role_id',
            relatedPivotKey: 'content_privilege_id'
        )->withPivot('pivot_id');
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function backofficePrivileges(): BelongsToMany
    {
        return $this->belongsToMany(
            related: BackofficePrivilege::class,
            table: BackofficePrivilege::PIVOT_ROLE_TABLE,
            foreignPivotKey: 'role_id',
            relatedPivotKey: 'backoffice_section_id',
        )->withPivot('pivot_id');
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function menuPrivileges(): BelongsToMany
    {
        return $this->belongsToMany(
            related: MenuPrivilege::class,
            table: MenuPrivilege::PIVOT_ROLE_TABLE,
            foreignPivotKey: 'role_id',
            relatedPivotKey: 'menu_privilege_id',
        )->withPivot('pivot_id');
    }

    /**
     * @mixin Builder
     * @return BelongsToMany
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'inside_users_roles', 'role_id', 'user_uuid');
    }

    /**
     * @mixin Builder
     * @return BelongsTo
     */
    public function category(): BelongsTo
    {
        return $this->belongsTo(RoleCategory::class, 'role_category_id');
    }

    /**
     * @return RolePrivilegesService
     */
    public function getAccessRestriction(): RolePrivilegesService
    {
        return RolePrivilegesService::of($this);
    }

    /**
     * @return void
     */
    public function computeRestriction(): void
    {
        ComputeRestrictionService::computeCategorizableCapabilitiesFor($this);
        ComputeRestrictionService::computeCapabilitiesFor($this);
        $this->getAccessRestriction()->clearCache('_'.$this->id);
    }

    /**
     * @param string $capability
     * @param Content|string $class
     * @return bool
     */
    public function hasContentTypePrivilegeTo(string $capability, Content|string $class): bool
    {
        return $this
            ->getAccessRestriction()
            ->getContentTypePrivileges()
            ->filter
            ->isAuthorized()
            ->map(fn (ContentTypePrivilegeDto $dto) => $dto->toArray())
            ->where('capability.name', $capability)
            ->where('type', $class instanceof Content ? $class::class : $class)
            ->isNotEmpty();
    }
}
