<?php

namespace Inside\Permission\Exodus\Http\Resources;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Actions\SortPrivileges\SortByDepth;
use Inside\Permission\Exodus\Dto\Privileges\MenuPrivilegeDto;
use Inside\Permission\Exodus\Enums\MenuEnum;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class MenuPrivilegeCollection
{
    private Collection $authenticated_privileges;

    private function __construct(
        private Collection $menuPrivileges,
        private string $locale,
        private bool $inheritance,
    ) {
        $this->authenticated_privileges = $this->inheritance
            ? RolePrivilegesService::getAuthenticatedMenuPrivilegesId()
            : collect();
    }

    public static function from(Collection $menuPrivileges, string $locale, bool $inheritance = false): self
    {
        return new self($menuPrivileges, $locale, $inheritance);
    }

    private function transform(): Collection
    {
        return $this->menuPrivileges
            ->map(fn (MenuPrivilegeDto $dto) => $this->flatten($dto))
            ->groupBy(['type', 'uuid'])
            ->map(fn (Collection $category) => $this->flattenByMenuCategory($category))
            ->mapWithKeys(function (Collection $type) {
                $menu = MenuEnum::label($type->first()['type'], $this->locale);
                $contents = (new SortByDepth())->execute($type)->sortBy('weight')->values();

                return [$menu => $contents];
            });
    }

    private function flatten(MenuPrivilegeDto $dto): array
    {
        return [
            'id' => $dto->getId(),
            'label' => $dto->getIndex()?->getMenu()?->getTitle(),
            'langcode' => $dto->getIndex()?->getMenu()?->getLangcode(),
            'uuid' => $dto->getIndex()?->getUuid(),
            'parent_uuid' => $dto->getIndex()?->getMenu()?->getParentUuid(),
            'type' => $dto->getIndex()?->getType(),
            'capability' => $dto->getCapability()->toArray(),
            'weight' => $dto->getIndex()?->getMenu()?->getWeight(),
            'is_authorized' => $dto->isAuthorized(),
            'is_inherited' => $this->authenticated_privileges->contains($dto->getId()),
        ];
    }

    private function flattenByMenuCategory(Collection $category): Collection
    {
        return $category->mapWithKeys(fn (Collection $content) => [
            $content->first()['uuid'] => [
                'label' => $content->first()['label'],
                'type' => $content->first()['type'],
                'uuid' => $content->first()['uuid'],
                'langcode' => $content->first()['langcode'],
                'weight' => $content->first()['weight'],
                'parent_uuid' => $content->first()['parent_uuid'],
                'privileges' => $content->map(fn (array $privilege) => [
                    'privilege_id' => $privilege['id'],
                    'capability_id' => $privilege['capability']['id'],
                    'capability' => $privilege['capability']['name'],
                    'is_authorized' => $privilege['is_authorized'],
                    'is_inherited' => $privilege['is_inherited'],
                ]),
                'children' => [],
            ],
        ])->sortBy('weight');
    }

    public function toArray(): array
    {
        return [
            'data' => $this->transform()->sortBy('label')->toArray(),
        ];
    }
}
