<?php

namespace Inside\Permission\Exodus\Http\Resources;

use Illuminate\Support\Collection;
use Inside\Content\Models\Content;
use Inside\Permission\Exodus\Dto\Privileges\ContentTypePrivilegeDto;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class ContentTypePrivilegeCollection
{
    private Collection $authenticated_privileges;

    private function __construct(
        private Collection $contentTypePrivileges,
        private string $locale,
        private bool $inheritance,
    ) {
        $this->authenticated_privileges = $this->inheritance
            ? RolePrivilegesService::getAuthenticatedContentTypePrivilegesId()
            : collect();
    }

    public static function from(Collection $contentTypePrivileges, string $locale, bool $inheritance = false): self
    {
        return new self($contentTypePrivileges, $locale, $inheritance);
    }

    private function transform(): Collection
    {
        return $this
            ->contentTypePrivileges
            ->map(fn (ContentTypePrivilegeDto $dto) => $this->flatten($dto))
            ->groupBy('name')
            ->map(fn (Collection $privileges, string $type) => $this->flattenByContentType($privileges, $type)->toArray());
    }

    private function flatten(ContentTypePrivilegeDto $dto): array
    {
        /** @var Content $class */
        $class = $dto->getType();

        return [
            'id' => $dto->getId(),
            'label' => $class::labels()->get($this->locale),
            'name' => $dto->getType(),
            'capability' => $dto->getCapability()->toArray(),
            'is_authorized' => $dto->isAuthorized(),
            'is_inherited' => $this->authenticated_privileges->contains($dto->getId()),
        ];
    }

    private function flattenByContentType(Collection $privileges, string $type): Collection
    {
        return collect([
            'label' => $privileges->first()['label'],
            'type' => $type,
            'is_categorizable' => $type::isCategorizable(),
            'privileges' => $privileges->map(fn (array $privilege) => [
                'privilege_id' => $privilege['id'],
                'capability_id' => $privilege['capability']['id'],
                'capability' => $privilege['capability']['name'],
                'is_authorized' => $privilege['is_authorized'],
                'is_inherited' => $privilege['is_inherited'],
            ]),
        ]);
    }

    public function toArray(): array
    {
        return [
            'data' => $this->transform()->sortBy('label')->values()->toArray(),
        ];
    }
}
