<?php

namespace Inside\Permission\Exodus\Http\Resources;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Actions\SortPrivileges\SortByDepth;
use Inside\Permission\Exodus\Dto\Privileges\ContentPrivilegeDto;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class CategorizableContentPrivilegeCollection
{
    private Collection $authenticated_privileges;

    private function __construct(
        private Collection $categorizableContentPrivileges,
        private string $locale,
        private bool $inheritance,
    ) {
        $this->authenticated_privileges = $this->inheritance
            ? RolePrivilegesService::getAuthenticatedCategorizableContentPrivilegesId()
            : collect();
    }

    public static function from(Collection $categorizableContentPrivileges, string $locale, bool $inheritance = false): self
    {
        return new self($categorizableContentPrivileges, $locale, $inheritance);
    }

    private function transform(): Collection
    {
        return $this->categorizableContentPrivileges
            ->map(fn (ContentPrivilegeDto $dto) => $this->flatten($dto))
            ->groupBy(['type', 'uuid'])
            ->map(fn (Collection $type) => $this->flattenByContent($type))
            ->mapWithKeys(function (Collection $type) {
                $category = $type->first()['type']::labels()->get($this->locale);
                $contents = (new SortByDepth())->execute($type)->sortBy('label')->values();

                return [$category => $contents];
            });
    }

    private function flatten(ContentPrivilegeDto $dto): array
    {
        $translation = collect($dto->getIndex()?->getTranslations()->map->toArray());

        $local = $translation->firstWhere('translatable.langcode', $this->locale);

        if (data_get($local, 'translatable.title') === null) {
            $local = $translation->firstWhere('translatable.title');
        }

        if (data_get($local, 'translatable.langcode') === $this->locale) {
            $label = data_get($local, 'translatable.title');
        } else {
            $label = sprintf('[%s] %s', strtoupper(data_get($local, 'translatable.langcode')), data_get($local, 'translatable.title'));
        }

        return $dto->collect()->merge([
            'label' => $label,
            'type' => $dto->getIndex()?->getType(),
            'uuid' => data_get($local, 'uuid'),
            'parent_uuid' => data_get($local, 'translatable.parent_uuid'),
            'uuid_host' => $dto->getIndex()?->getUuidHost(),
            'status' => data_get($local, 'translatable.status'),
            'link' => data_get($local, 'translatable.link'),
        ])->forget('index')->toArray();
    }

    private function flattenByContent(Collection $type): Collection
    {
        return $type->mapWithKeys(fn (Collection $content) => [$content->first()['uuid'] => [
            'label' => $content->first()['label'],
            'type' => $content->first()['type'],
            'uuid' => $content->first()['uuid'],
            'parent_uuid' => $content->first()['parent_uuid'],
            'uuid_host' => $content->first()['uuid_host'],
            'status' => $content->first()['status'],
            'link' => $content->first()['link'],
            'privileges' => $content->map(fn (array $privilege) => [
                'privilege_id' => $privilege['id'],
                'capability_id' => $privilege['capability']['id'],
                'capability' => $privilege['capability']['name'],
                'is_authorized' => $privilege['is_authorized'],
                'is_inherited' => $this->authenticated_privileges->contains($privilege['id']),
            ]),
            'children' => [],
        ]]);
    }

    public function toArray(): array
    {
        return [
            'data' => $this->transform()->sortKeys()->toArray(),
        ];
    }
}
