<?php

namespace Inside\Permission\Exodus\Http\Resources;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Dto\Privileges\BackofficePrivilegeDto;
use Inside\Permission\Exodus\Enums\BackofficeEnum;
use Inside\Permission\Exodus\Models\Role;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

class BackofficePrivilegeCollection
{
    private Collection $authenticated_privileges;

    private function __construct(
        private Collection $backofficePrivileges,
        private string $locale,
        private bool $inheritance,
    ) {
        $this->authenticated_privileges = $this->inheritance
            ? RolePrivilegesService::getAuthenticatedBackofficePrivilegesId()
            : collect();
    }

    public static function from(Collection $backofficePrivileges, string $locale, bool $inheritance = false): self
    {
        return new self($backofficePrivileges, $locale, $inheritance);
    }

    private function transform(): Collection
    {
        return $this->backofficePrivileges
            ->map(fn (BackofficePrivilegeDto $dto) => $this->flatten($dto))
            ->groupBy('name')
            ->map(fn (Collection $backoffice) => $this->flattenByBackoffice($backoffice)->toArray());
    }

    private function flatten(BackofficePrivilegeDto $dto): array
    {
        return [
            'id' => $dto->getId(),
            'label' => BackofficeEnum::label($dto->getName(), $this->locale),
            'name' => $dto->getName(),
            'capability' => $dto->getCapability()->toArray(),
            'is_authorized' => $dto->isAuthorized(),
            'is_inherited' => $this->authenticated_privileges->contains($dto->getId()),
        ];
    }

    private function flattenByBackoffice(Collection $backoffice): Collection
    {
        return collect([
            'label' => $backoffice->first()['label'],
            'name' => $backoffice->first()['name'],
            'privileges' => $backoffice->map(fn (array $privilege) => [
                'privilege_id' => $privilege['id'],
                'capability_id' => $privilege['capability']['id'],
                'capability' => $privilege['capability']['name'],
                'is_authorized' => $privilege['is_authorized'],
                'is_inherited' => $privilege['is_inherited'],
            ]),
        ]);
    }

    public function toArray(): array
    {
        return [
            'data' => $this->transform()->sortBy('label')->values()->toArray(),
        ];
    }
}
