<?php

namespace Inside\Permission\Exodus\Actions\StorePrivileges;

use Illuminate\Support\Collection;
use Inside\Permission\Exodus\Actions\RemoveDuplicateAuthenticatedPrivileges\RemoveDuplicateAuthenticatedPrivilegesContentType;
use Inside\Permission\Exodus\Dto\CapabilityDto;
use Inside\Permission\Exodus\Dto\Privileges\ContentTypePrivilegeDto;
use Inside\Permission\Exodus\Models\Role;

class StoreContentTypePrivileges
{
    public function execute(Role $role, array $data): void
    {
        // Clear inside cache
        $role->getAccessRestriction()->clearCache('content_type');

        $privileges = collect($data)
            ->map(fn (array $type) => $this->extractPrivileges(collect($type)))
            ->flatten(1)
            ->filter(fn (ContentTypePrivilegeDto $dto) => $dto->isAuthorized())
            ->map(fn (ContentTypePrivilegeDto $dto) => $dto->getId())
            ->values();

        // Sync privileges
        $role->contentTypePrivileges()->sync($privileges);

        // Remove duplicate authenticated privileges
        if ($role->name === Role::AUTHENTICATED) {
            (new RemoveDuplicateAuthenticatedPrivilegesContentType())->execute();
        }

        // Recompute view restriction
        $role->computeRestriction();
    }

    private function extractPrivileges(Collection $type): array
    {
        return collect($type->get('privileges'))
            ->map(fn (array $privilege) => ContentTypePrivilegeDto::from(
                $type['type'],
                CapabilityDto::from($privilege['capability'], $privilege['capability_id']),
                $privilege['privilege_id'],
            )->setAuthorization($privilege['is_inherited'] ? false : $privilege['is_authorized']))
            ->toArray();
    }
}
