<?php

namespace Inside\Permission\Exodus\Actions\SortPrivileges;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;

class SortByDepth
{
    public function execute(Collection $privileges, string $key = 'parent_uuid', bool $keepOrphan = false): Collection
    {
        /**
         * Checks if all items in the collection have a parent_uuid that is not null.
         * If all items have a null parent_uuid, it means they are all root items, so the function simply returns the collection as it is.
         * If there are items with a non-null parent_uuid, it means there are child items in the collection.
         */
        if ($privileges->where($key, '<>', null)->isEmpty()) {
            return $privileges->values();
        }

        $alterable = $privileges->toArray();
        $alreadySorted = [];

        /**
         *  For each of these items, it checks if there are any other items in the collection that have this item as their parent
         *  If there are no such items, it means this item is a leaf node (an item with no children).
         *  The function then adds this item to the children array of its parent item.
         *  Also adds its uuid to the $alreadySorted array to keep track of items that have been sorted.
         */
        $privileges->where($key, '<>', null)->each(function (array $content) use ($privileges, &$alterable, &$alreadySorted, $key, $keepOrphan) {
            if ($privileges->where('uuid', $content[$key])->isEmpty()) {
                if (! $keepOrphan) {
                    /**
                     * If the parent_uuid of an item is not found in the collection, it means the parent item is not present in the collection.
                     */
                    Log::error(sprintf('[Inside - Permission] Parent not found for (%s) %s : %s', $content['type'], $content['uuid'], $content['label']));

                    $alreadySorted[] = $content['uuid'];
                } else {
                    $alterable[$content['uuid']][$key] = null;
                }
            } elseif ($privileges->where($key, $content['uuid'])->isEmpty()) {
                /**
                 * If the content has no children, it is added to the $alterable array and its uuid is added to the $alreadySorted array.
                 */
                $alterable[$content[$key]]['children'][] = $content;
                $alreadySorted[] = $content['uuid'];
            } elseif ($content[$key] === $content['uuid']) {
                /**
                 * If the parent_uuid of an item is the same as its uuid, it means there is a circular reference.
                 */
                Log::error(sprintf('[Inside - Permission] Circular reference detected for (%s) %s : %s', $content['type'], $content['uuid'], $content['label']));

                $alreadySorted[] = $content['uuid'];
            }
        });

        /**
         * Sorting the children
         */
        $alterableCollection = collect($alterable);
        $alterableCollection = $alterableCollection->map(function ($item) {
            // Check if the item has children and if it's an array
            if (isset($item['children']) && is_array($item['children'])) {
                // Convert children to a collection, sort by 'weight', and convert back to an array
                $item['children'] = collect($item['children'])->sortBy(fn ($child) => $child['weight'] ?? 0)
                    ->values()
                    ->all();
            }

            return $item;
        });

        /**
         * After all items with a non-null parent_uuid have been processed, the function calls itself recursively
         * with a new collection that excludes the items that have been already sorted.
         */
        return $this->execute($alterableCollection->whereNotIn('uuid', $alreadySorted), $key, $keepOrphan);
    }
}
