<?php

namespace Inside\Permission\Exodus\Actions\RemoveDuplicateAuthenticatedPrivileges;

use Illuminate\Support\Facades\DB;
use Inside\Permission\Exodus\Enums\CapabilityEnum;
use Inside\Permission\Exodus\Models\Capability;
use Inside\Permission\Exodus\Models\Privileges\CategorizableContentPrivilege;
use Inside\Permission\Exodus\Models\ViewModels\CategorizableContentIndex;
use Inside\Permission\Exodus\Services\RolePrivilegesService;

final class RemoveDuplicateAuthenticatedPrivilegesCategorizable
{
    public function execute(): int
    {
        $authenticated_id = RolePrivilegesService::getAuthenticated()->id;
        $authenticated_privileges = RolePrivilegesService::getAuthenticatedCategorizableContentPrivilegesId();

        /**
         * We get the duplicated privileges already owned by Authenticated role.
         * So it's not necessary to keep them in other roles.
         */
        $duplicated_privileges = DB::table(CategorizableContentPrivilege::PIVOT_ROLE_TABLE)
            ->join(CategorizableContentPrivilege::TABLE, CategorizableContentPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', CategorizableContentPrivilege::TABLE.'.id')
            ->where('role_id', '!=', $authenticated_id)
            ->whereIn('content_privilege_id', $authenticated_privileges)
            ->get();

        /**
         * But in privilege, we doesn't want to remove the READ capability, if the role has another capability on the same categorizable content type.
         * In the future, if the role Authenticated lost the READ capability, it will be lost for all roles.
         * So you could get an Assign / Edit / Delete capability without the READ capability.
         * That why in the next query, we exclude the READ capability and delete all other capabilities.
         */
        $advanced_privileges = DB::table(CategorizableContentPrivilege::PIVOT_ROLE_TABLE)
            ->join(CategorizableContentPrivilege::TABLE, CategorizableContentPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', CategorizableContentPrivilege::TABLE.'.id')
            ->join(Capability::TABLE, CategorizableContentPrivilege::TABLE.'.capability_id', Capability::TABLE.'.id')
            ->whereIn('pivot_id', $duplicated_privileges->pluck('pivot_id'))
            ->where('name', '<>', CapabilityEnum::READ)
            ->pluck('pivot_id');

        $count = DB::table(CategorizableContentPrivilege::PIVOT_ROLE_TABLE)
            ->whereIn('pivot_id', $advanced_privileges)
            ->delete();

        /**
         * For each Role, and each Content uuid_host, we count the total of privileges.
         */
        $all_privileges = DB::table(CategorizableContentPrivilege::PIVOT_ROLE_TABLE)
            ->join(CategorizableContentPrivilege::TABLE, CategorizableContentPrivilege::PIVOT_ROLE_TABLE.'.content_privilege_id', CategorizableContentPrivilege::TABLE.'.id')
            ->join(Capability::TABLE, CategorizableContentPrivilege::TABLE.'.capability_id', Capability::TABLE.'.id')
            ->selectRaw('role_id, uuid_host, COUNT(*) as total_privileges')
            ->groupBy('role_id', 'uuid_host')
            ->get();

        /**
         *  It's time to take a closer look at VIEW capability.
         *  We want to delete privilege if the role has only the VIEW capability.
         */
        $view_privileges = $duplicated_privileges
            ->whereNotIn('pivot_id', $advanced_privileges)
            ->filter(fn ($privilege) => $all_privileges->where('role_id', $privilege->role_id)->where('uuid_host', $privilege->uuid_host)->first()?->total_privileges === 1)
            ->pluck('pivot_id');

        $count += DB::table(CategorizableContentPrivilege::PIVOT_ROLE_TABLE)
            ->whereIn('pivot_id', $view_privileges)
            ->delete();

        /**
         * We return the total of deleted privileges.
         */
        return $count;
    }
}
